package main

import (
	"net"
	"os"
	"os/signal"
	"strings"
	"time"

	"code.justin.tv/feeds/dnscache"
	"code.justin.tv/feeds/ecs-on-demand/cmd/ecs-on-demand/internal/ondemand"
	"code.justin.tv/feeds/service-common"
	xraydynamo "code.justin.tv/feeds/xray/plugins/dynamodb"
	"code.justin.tv/feeds/xray/plugins/ec2"
	"github.com/aws/aws-sdk-go/service/dynamodb"
)

const (
	teamName    = "feeds"
	serviceName = "ondemand"
)

var instance = service{
	osExit: os.Exit,
	serviceCommon: service_common.ServiceCommon{
		ConfigCommon: service_common.ConfigCommon{
			Team:       teamName,
			Service:    serviceName,
			OsGetenv:   os.Getenv,
			OsHostname: os.Hostname,
		},
	},
}

type service struct {
	osExit        func(code int)
	onListen      func(net.Addr)
	sigChan       chan os.Signal
	serviceCommon service_common.ServiceCommon
	runner        service_common.ServiceRunner

	server ondemand.HTTPServer

	configs struct {
		HTTPConfig         service_common.BaseHTTPServerConfig
		taskDBConfig       ondemand.TaskDBConfig
		taskCreationConfig ondemand.TaskCreationConfig
	}
}

func (f *service) setup() error {
	if err := f.serviceCommon.Setup(); err != nil {
		return err
	}
	if err := service_common.LoadConfigs(f.serviceCommon.Config, &f.configs.taskDBConfig, &f.configs.taskCreationConfig); err != nil {
		return err
	}
	if err := f.configs.HTTPConfig.Verify(f.serviceCommon.Config, "ondemand"); err != nil {
		return err
	}

	if err := f.serviceCommon.XRay.WithPlugin(&ec2.Plugin{}); err != nil {
		f.serviceCommon.Log.Log("err", err, "unable to enable ec2 plugin")
	}
	if err := f.serviceCommon.XRay.WithPlugin(&xraydynamo.Plugin{}); err != nil {
		f.serviceCommon.Log.Log("err", err, "unable to enable ec2 plugin")
	}

	f.serviceCommon.ConfigCommon.DNSCache = dnscache.Cache{
		OnErr: func(err error, host string) {
			f.serviceCommon.Log.Log("host", host, "err", err, "unable to use DNS cache")
		},
		OnCacheUpsert: func(host string, oldAddrs []string, newAddrs []string, lookupDuration time.Duration) {
			oldIPs := strings.Join(oldAddrs, ",")
			newIPs := strings.Join(newAddrs, ",")
			f.serviceCommon.Log.Debug("host", host, "oldIPs", oldIPs, "newIPs", newIPs, "lookupDuration", lookupDuration, "updated DNS cache")
		},
	}
	return nil
}

func (f *service) inject() {
	tr := service_common.DefaultTransport()
	tr.DialContext = f.serviceCommon.ConfigCommon.DNSCache.DialContext
	awsSession, cfg := service_common.CreateAWSSession(f.serviceCommon.Config)
	dynamoClient := dynamodb.New(awsSession, cfg...)
	TaskDB := &ondemand.TaskDB{
		DynamoClient: dynamoClient,
		Config:       &f.configs.taskDBConfig,
	}

	f.server = ondemand.HTTPServer{
		TaskDB: TaskDB,
		BaseHTTPServer: service_common.BaseHTTPServer{
			Config: &f.configs.HTTPConfig,
			Stats: &service_common.StatSender{
				SubStatter:   f.serviceCommon.Statsd.NewSubStatter("http"),
				ErrorTracker: &f.serviceCommon.ErrorTracker,
			},
			Dims:        &f.serviceCommon.CtxDimensions,
			Log:         f.serviceCommon.Log,
			ElevateKey:  f.serviceCommon.ElevateLogKey,
			Ctxlog:      &f.serviceCommon.Ctxlog,
			OnListen:    f.onListen,
			PanicLogger: f.serviceCommon.PanicLogger,
			XRay:        f.serviceCommon.XRay,
		},
		TaskCreation: ondemand.TaskCreation{
			TaskCreationConfig: &f.configs.taskCreationConfig,
			DeployDB:           &ondemand.DeployDB{},
			DockerHelper: &ondemand.DockerHelper{
				DockerHost: "docker.internal.justin.tv",
			},
			TaskDB: TaskDB,
		},
	}
	f.server.SetupRoutes = f.server.Routes

	f.runner = service_common.ServiceRunner{
		Log: f.serviceCommon.Log,
		Services: []service_common.Service{
			&f.server, &f.serviceCommon, &f.server.TaskCreation,
		},
		SigChan:      f.sigChan,
		SignalNotify: signal.Notify,
	}

	// Cannot just pass f because f contains private members that I cannot nil check via reflection
	res := (&service_common.NilCheck{
		IgnoredPackages: []string{"aws-sdk-go", "net/http"},
	}).Check(f, &f.server, f.configs, f.runner)
	res.MustBeEmpty(os.Stderr)
}

func (f *service) main() {
	if err := f.setup(); err != nil {
		if f.serviceCommon.Log != nil {
			f.serviceCommon.Log.Log("err", err, "Unable to load initial config")
		}
		service_common.SetupLogger.Log("err", err, "Unable to load initial config")
		time.Sleep(time.Millisecond * 150)
		f.osExit(1)
		return
	}
	f.inject()
	if err := f.runner.Execute(); err != nil {
		f.serviceCommon.Log.Log("err", err, "wait to end finished with an error")
		time.Sleep(time.Millisecond * 150)
		f.osExit(1)
		return
	}
	f.serviceCommon.Log.Log("Finished main")
}

func main() {
	instance.main()
}
