package xray

import (
	"io"
	"sync"
	"sync/atomic"

	"github.com/aws/aws-sdk-go/aws/request"
)

// Logger can optionally be added to XRay to log out error conditions while XRay is operating
type Logger interface {
	OnError(msg string, err error)
	Msg(msg string)
}

// RandSource is how xray picks random numbers for sampling percentages.  It fits the math.Rand interface, but beware
// your implementation of Float64() must be thread safe.
type RandSource interface {
	// Float64 should return a value [0-1]
	Float64() float64
}

// XRay is the instance that allows sending xray information to the locally running agent
type XRay struct {
	ErrStacks        []StackFramesFromError
	optionalMetadata OptionalMetadata
	Logger           Logger
	RandSource       RandSource

	emitterLock sync.Mutex
	emitter     io.Writer

	configMutex           sync.Mutex
	storedConfig          *Config
	onceLoadDefaultConfig sync.Once

	xrayStats xrayStats
}

type xrayStats struct {
	TotalSegments int64
	EmitAttempts  int64
	EmittedTotal  int64
	TotalErrors   int64
}

func (x *XRay) onError(msg string, err error) {
	atomic.AddInt64(&x.xrayStats.TotalErrors, 1)
	if x.Logger != nil {
		x.Logger.OnError(msg, err)
	}
}

func (x *XRay) msg(msg string) {
	if x.Logger != nil {
		x.Logger.Msg(msg)
	}
}

// StackFramesFromError allows users to extract stack frames from go's error interface if their system wraps errors
// with something that contains a stack trace
type StackFramesFromError interface {
	StackTrace(err error) []uintptr
}

func (x *XRay) config() *Config {
	// This once flag is a fast atomic() check, but allows people to use XRay
	// struct directly
	x.onceLoadDefaultConfig.Do(func() {
		if x.storedConfig == nil {
			if err := x.Configure(Config{}); err != nil {
				x.onError("unable to setup initial configuration", err)
			}
		}
	})
	x.configMutex.Lock()
	ret := x.storedConfig
	x.configMutex.Unlock()
	return ret
}

// Plugin can be added to XRay to modify metadata attached to every segment
type Plugin interface {
	ModifyMetadata(meta *OptionalMetadata) error
}

// WithPlugin attaches a plugin to XRay
func (x *XRay) WithPlugin(p Plugin) error {
	return p.ModifyMetadata(&x.optionalMetadata)
}

// OptionalMetadata is optional information attached to each segment.  Modify this via Plugins
type OptionalMetadata struct {
	IdentityDocument  *IdentityDocument
	BeanstalkMetadata *BeanstalkMetadata
	ECSContainerName  string
	// returns the dynamodb table name from a request, if one is in there
	DynamoDBTableDetector func(r *request.Request) string
}

// IdentityDocument includes EC2 identity information
type IdentityDocument struct {
	AccountID        string
	InstanceID       string
	AvailabilityZone string
}

// BeanstalkMetadata includes beanstalk metadata
type BeanstalkMetadata struct {
	Environment  string `json:"environment_name"`
	VersionLabel string `json:"version_label"`
	DeploymentID int    `json:"deployment_id"`
}
