package sandyconf

import (
	"time"

	"code.justin.tv/systems/sandstorm/manager"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sts"
)

const (
	defaultDatacenter = "us-west-2"
)

// ManagerConstructor provides an easy way to get a sandstorm manager with reasonable defaults.
type ManagerConstructor struct {
	AwsConfig *aws.Config
	// Deprecated: ignored
	TableName string
	// Deprecated: ignored
	KeyID string
}

func (m *ManagerConstructor) region() *string {
	if m.AwsConfig == nil || m.AwsConfig.Region == nil {
		return aws.String(defaultDatacenter)
	}
	return m.AwsConfig.Region
}

// Region sets the region to use for the constructor
func (m *ManagerConstructor) Region(region string) *ManagerConstructor {
	if m.AwsConfig == nil {
		m.AwsConfig = &aws.Config{}
	}
	m.AwsConfig.Region = &region
	return m
}

// Manager creates a manager that assumes an ARN role
func (m *ManagerConstructor) Manager(roleARN string) (*manager.Manager, error) {
	awsConfig := &aws.Config{}
	awsConfig.MergeIn(m.AwsConfig)
	awsConfig.Region = m.region()
	awsSession, err := session.NewSession(m.AwsConfig)
	if err != nil {
		return nil, err
	}
	stsclient := sts.New(awsSession)
	arp := &stscreds.AssumeRoleProvider{
		ExpiryWindow: 10 * time.Second,
		RoleARN:      roleARN,
		Client:       stsclient,
	}
	manager.DefaultConfig()
	awsCreds := credentials.NewCredentials(arp)
	awsConfig.WithCredentials(awsCreds)
	mgr := manager.New(manager.Config{
		AWSConfig: awsConfig,
	})
	return mgr, nil
}
