// +build integration

package agent

import (
	"errors"
	"fmt"
	"io/ioutil"
	"os"
	"path"
	"testing"
	"time"

	"code.justin.tv/systems/sandstorm/inventory/consumedsecrets"
	"code.justin.tv/systems/sandstorm/manager"
	mgrMocks "code.justin.tv/systems/sandstorm/manager/mocks"
	"code.justin.tv/systems/sandstorm/mocks"
	"code.justin.tv/systems/sandstorm/queue"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/service/sqs"
	"github.com/cenkalti/backoff"
	uuid "github.com/satori/go.uuid"
	. "github.com/smartystreets/goconvey/convey"
	"github.com/stretchr/testify/mock"
)

func TestAgentBasic(t *testing.T) {

	Convey("For a configured Agent", t, func() {

		testFolderPath := prepareTestPath("agent-basic-test")
		agent := getTestAgent(testFolderPath, true, false)

		So(agent.State.Templates, ShouldNotBeEmpty)

		Convey("->isTemplateWhitelisted()", func() {

			filterTestSource := testFolderPath + "/filter_test"

			template := &Template{
				Source:      filterTestSource,
				Destination: testFolderPath + "/output/filter_test",
			}

			Convey("should always return true if whitelist is empty", func() {
				So(agent.isTemplateWhitelisted(template), ShouldBeTrue)
			})

			Convey("should whitelist properly if list is not empty", func() {
				agent.TemplateWhitelist = []string{testFolderPath, "/other_path"}
				So(agent.isTemplateWhitelisted(template), ShouldBeFalse)

				agent.TemplateWhitelist = append(agent.TemplateWhitelist, filterTestSource)
				So(agent.isTemplateWhitelisted(template), ShouldBeTrue)
			})
		})

		Convey("->addTemplate()", func() {

			Convey("Should add a template with a unique src/dest", func() {

				templateCount := len(agent.State.Templates)

				template := &Template{
					Source:      testFolderPath + "/super_unique",
					Destination: testFolderPath + "/output/super_unique",
				}

				err := agent.AddTemplate(template)
				So(err, ShouldBeNil)
				So(len(agent.State.Templates), ShouldEqual, templateCount+1)
			})

			Convey("Should fail if", func() {

				existingSrc := "uniqueSrc"
				existingDest := "uniqueDest"
				baseTemplate := &Template{
					Source:      existingSrc,
					Destination: existingDest,
				}

				insertErr := agent.AddTemplate(baseTemplate)
				So(insertErr, ShouldBeNil)

				Convey("a template source isn't unique", func() {
					template := &Template{
						Source:      existingSrc,
						Destination: testFolderPath + "/output/unique123",
					}

					err := agent.AddTemplate(template)
					So(err, ShouldNotBeNil)
				})

				Convey("a template destination isn't unique", func() {
					template := &Template{
						Source:      testFolderPath + "/unique456",
						Destination: existingDest,
					}

					err := agent.AddTemplate(template)
					So(err, ShouldNotBeNil)
				})

				Convey("a template src/dest are the same", func() {
					template := &Template{
						Source:      testFolderPath + "/unique789",
						Destination: testFolderPath + "/unique789",
					}

					err := agent.AddTemplate(template)
					So(err, ShouldNotBeNil)
				})
			})
		})

		Convey("->fetchSecretsByName()", func() {

			Convey("should successfully get secrets from KMS", func() {

				names := []string{
					"systems/sandstorm-agent/development/test_secret",
					"systems/sandstorm-agent/development/test_secret_2",
				}

				secrets, err := agent.FetchSecretsByName(names)
				So(err, ShouldBeNil)
				So(len(secrets), ShouldEqual, 2)

				So(secrets[0].Name, ShouldEqual, names[0])
				So(secrets[1].Name, ShouldEqual, names[1])
			})
		})

		Convey("->getEffectedTemplates()", func() {
			secrets, err := agent.FetchSecretsByName([]string{"systems/sandstorm-agent/development/test_secret"})
			So(err, ShouldBeNil)

			secrets[0].UpdatedAt++
			secrets[0].Plaintext = []byte("updated")

			Convey("should return all templates containing the secret", func() {
				templates := agent.getEffectedTemplates(secrets)
				So(len(templates), ShouldEqual, 2)
			})

			Convey("should return only whitelisted templates containing the secret", func() {
				agent.TemplateWhitelist = []string{path.Join(getTestConfigPath(), "/templates.d/sample_template")}

				templates := agent.getEffectedTemplates(secrets)
				So(len(templates), ShouldEqual, 1)
				So(templates[0].Source, ShouldEqual, agent.TemplateWhitelist[0])
			})
		})

		Convey("->writeStateFile()", func() {

			stateFilePath := testFolderPath + "/writeStateFile-test"
			err := agent.writeStateFile(stateFilePath)
			So(err, ShouldBeNil)

			So(fileExists(stateFilePath), ShouldBeTrue)
		})

		Convey("->updateState()", func() {

			testSecret := &manager.Secret{
				Name:      "systems/sandstorm-agent/development/test_secret",
				UpdatedAt: 1000,
				Plaintext: []byte("defaultPlaintext"),
			}

			Convey("should replace an older secret version", func() {
				// set state
				for _, template := range agent.State.Templates {
					if _, exists := template.Secrets[testSecret.Name]; exists {
						template.Secrets[testSecret.Name] = testSecret
						template.Dirty = false
					}
				}

				newPlaintext := []byte("foobar")
				newerSecret := &manager.Secret{}
				*newerSecret = *testSecret
				newerSecret.UpdatedAt = 1001
				newerSecret.Plaintext = newPlaintext

				secrets := []*manager.Secret{newerSecret}

				Convey("should not update any templates if none are provided", func() {

					agent.updateState(nil, []*manager.Secret{newerSecret})

					for _, template := range agent.State.Templates {
						secret, exists := template.Secrets[testSecret.Name]
						if exists {
							So(secret, ShouldEqual, testSecret)
							So(template.Dirty, ShouldBeFalse)
						}
					}
				})

				Convey("should update state templates if provided", func() {
					templates := agent.getEffectedTemplates(secrets)

					agent.updateState(templates, []*manager.Secret{newerSecret})

					changeCount := 0
					for _, template := range agent.State.Templates {
						secret, exists := template.Secrets[testSecret.Name]
						if exists {
							So(secret, ShouldEqual, newerSecret)
							So(template.Dirty, ShouldBeTrue)
							changeCount++
						}
					}
					So(changeCount, ShouldEqual, len(templates))
				})

				Convey("should initialize template secrets loaded from statefile", func() {
					stateFilePath := path.Join(testFolderPath, "updateStateStateFile")
					err := agent.writeStateFile(stateFilePath)
					So(err, ShouldBeNil)

					err = syncWithStateFile(stateFilePath, agent.State.Templates)
					So(err, ShouldBeNil)

					populatedSecret := &manager.Secret{}
					*populatedSecret = *testSecret
					populatedSecret.Plaintext = []byte("not empty")
					populatedSecret.UpdatedAt = 1002

					secrets := []*manager.Secret{populatedSecret}
					templates := agent.getEffectedTemplates(secrets)
					So(templates, ShouldNotBeEmpty)

					agent.updateState(templates, secrets)

					for _, template := range agent.State.Templates {
						secret, exists := template.Secrets[testSecret.Name]
						if exists {
							So(secret, ShouldEqual, populatedSecret)
							So(template.Dirty, ShouldBeTrue)
						}
					}
				})
			})
		})

		Convey("->outputTemplates()", func() {

			templates := []*Template{}
			for _, template := range agent.State.Templates {
				template.Dirty = true
				templates = append(templates, template)
			}

			Convey("should output all dirty templates", func() {

				successfulTemplates, err := outputTemplates(templates)
				So(err, ShouldBeNil)
				So(len(successfulTemplates), ShouldEqual, len(templates))

				for _, template := range agent.State.Templates {
					So(fileExists(template.Destination), ShouldBeTrue)
					So(template.Dirty, ShouldBeFalse)
				}
			})
		})

		Convey("->runRestartCommands()", func() {
			restartPath := prepareTestPath("agent-runrestart-test")

			touchFilePath := path.Join(restartPath, "test-file")
			template := &Template{
				Command:       fmt.Sprintf("touch %s", touchFilePath),
				CommandStatus: &CommandStatus{},
			}

			err := runRestartCommands([]*Template{template}, 1)
			So(err, ShouldBeNil)
			So(fileExists(touchFilePath), ShouldBeTrue)
		})
	})
}

func TestAgentPolling(t *testing.T) {

	Convey("Test Agent Polling", t, func() {

		testFolderPath := prepareTestPath("agent-polling-test")
		agent := getTestAgent(testFolderPath, true, false)
		mockedSQS := new(mocks.SQSAPI)
		agent.Queue = &queue.Queue{
			SQS: mockedSQS,
		}

		Convey("->pollForSecrets()", func() {
			Convey("should return a list of updated secrets", func() {
				whitelist := []string{
					"systems/sandstorm-agent/development/poll_test_",
				}
				mockedOutput := &sqs.ReceiveMessageOutput{
					Messages: []*sqs.Message{
						&sqs.Message{
							Body:          aws.String(sampleQueueMessage),
							ReceiptHandle: aws.String(testReceiptHandle),
						},
					},
				}
				emptyOutput := &sqs.ReceiveMessageOutput{
					Messages: []*sqs.Message{},
				}
				mockedSQS.On("ReceiveMessageWithContext", mock.Anything, mock.Anything, mock.Anything).Return(mockedOutput, nil).Once()
				mockedSQS.On("ReceiveMessageWithContext", mock.Anything, mock.Anything, mock.Anything).Return(emptyOutput, nil)
				mockedSQS.On("DeleteMessage", mock.Anything).Return(nil, nil)
				agent.Dwell = 4
				secrets := agent.pollForSecrets(whitelist)
				So(len(secrets), ShouldEqual, 1)
			})
		})
	})
}

func TestAgentRun(t *testing.T) {
	Convey("initial agent.run() tests", t, func() {

		testFolderPath := prepareTestPath("agent-run-test")
		agent := getTestAgent(testFolderPath, true, false)

		cs, err := getTestConsumedSecretsClient()
		So(err, ShouldBeNil)

		Convey("Should successfully output configs, run commands and heartbeat out", func() {

			err := agent.Run()
			So(err, ShouldBeNil)
			checkDestinationTemplates(agent)

			secretName := "systems/sandstorm-agent/development/test_secret"
			secrets := make([]*consumedsecrets.ConsumedSecret, 0)
			err = cs.IterateBySecret(secretName, func(cs *consumedsecrets.ConsumedSecret) (err error) {
				secrets = append(secrets, cs)
				return
			})
			So(err, ShouldBeNil)
			if err != nil {
				return
			}

			So(len(secrets), ShouldBeGreaterThan, 0)
			if len(secrets) == 0 {
				return
			}

			secret := secrets[0]
			So(secret.FirstRetrievedAt, ShouldBeGreaterThan, 0)
		})
	})
}

func TestAgentSync(t *testing.T) {

	Convey("initial agent.sync() tests", t, func() {

		testFolderPath := prepareTestPath("agent-sync-test")
		agent := getTestAgent(testFolderPath, true, false)
		secretNames := agent.GetAllSecretNames()

		Convey("Should successfully output configs and run commands", func() {

			err := agent.sync(secretNames, false, true, agent.Splay)
			So(err, ShouldBeNil)
			testPostRunState(agent, testFolderPath, false)

			Convey("and not run restart commands for a following unchanged run", func() {

				touchPath := path.Join(testFolderPath, "touchMe")
				err := os.Remove(touchPath)
				So(err, ShouldBeNil)

				err = agent.Start(false, true)
				So(err, ShouldBeNil)

				testPostRunState(agent, testFolderPath, true)
				So(fileExists(touchPath), ShouldBeFalse)
			})
		})

		Convey("Should successfully output configs and not run commands if disabled", func() {
			err := agent.sync(secretNames, true, true, agent.Splay)
			So(err, ShouldBeNil)

			testPostRunState(agent, testFolderPath, true)
		})

		Convey("Should successfully output configs and not run commands and not create state file", func() {
			err := agent.sync(secretNames, false, false, agent.Splay)
			So(err, ShouldBeNil)

			checkDestinationTemplates(agent)
			stateFilePath := path.Join(getTestConfigPath(), StateFileName)
			So(fileExists(stateFilePath), ShouldBeFalse)
		})

		Convey("Should populate templates/secrets loaded from statefile", func() {

			err := createStateFileFromAgent(agent, testFolderPath)
			So(err, ShouldBeNil)

			agent := getTestAgent(testFolderPath, false, false)
			err = agent.sync(secretNames, false, false, agent.Splay)
			So(err, ShouldBeNil)

			for _, template := range agent.State.Templates {
				for _, secret := range template.Secrets {
					So(secret.Plaintext, ShouldNotBeNil)
				}

				So(template.Dirty, ShouldBeFalse)
			}
		})

		Convey("Should block on blocked secrets", func(c C) {
			missingSecret := uuid.NewV4().String()
			updateMsg := newTestQueueMessage(missingSecret, time.Now())

			agent := getTestAgent(testFolderPath, false, false)
			mockSQS := new(mocks.SQSAPI)

			agent.Queue = &queue.Queue{
				SQS: mockSQS,
			}

			So(len(agent.State.BlockedSecrets), ShouldEqual, 0)

			testSecret := &manager.Secret{
				Name:      missingSecret,
				Plaintext: []byte(missingSecret),
			}
			createTestSecrets(agent.manager, testSecret)
			mockedOutput := &sqs.ReceiveMessageOutput{
				Messages: []*sqs.Message{
					&sqs.Message{
						Body:          aws.String(updateMsg),
						ReceiptHandle: aws.String(testReceiptHandle),
					},
				},
			}
			mockSQS.On("ReceiveMessageWithContext", mock.Anything, mock.Anything, mock.Anything).Return(mockedOutput, nil).Run(func(args mock.Arguments) {
				time.Sleep(3 * time.Second)
			})
			mockSQS.On("DeleteMessage", mock.Anything).Return(nil, nil).Once()

			syncErr := agent.sync([]string{missingSecret}, true, true, agent.Splay)
			So(syncErr, ShouldBeNil)
			So(len(agent.State.BlockedSecrets), ShouldEqual, 0)
		})

		Convey("Should only sync whitelisted templates", func() {
			agent.TemplateWhitelist = append(agent.TemplateWhitelist, "/build/go/src/code.justin.tv/systems/sandstorm/agent/test/config/templates.d/sample_template")
			cleanAgentFiles(testFolderPath)
			err := agent.sync(secretNames, true, false, agent.Splay)
			So(err, ShouldBeNil)
			So(fileExists(path.Join(testFolderPath, "sample_template2")), ShouldBeFalse)
			So(fileExists(path.Join(testFolderPath, "sample_template")), ShouldBeTrue)
		})
	})
}

func TestAgentStart(t *testing.T) {

	Convey("->Start()", t, func() {

		testFolderPath := prepareTestPath("agent-start-test")
		agent := getTestAgent(testFolderPath, true, false)

		Convey("should not error", func() {
			err := agent.Start(false, true)
			So(err, ShouldBeNil)
		})
	})
}

func TestAgentRunOnUpdate(t *testing.T) {

	Convey("->runOnUpdate", t, func() {

		testFolderPath := prepareTestPath("agent-runOnUpdate-test")
		agent := getTestAgent(testFolderPath, true, false)

		err := agent.Start(false, true)
		So(err, ShouldBeNil)

		Convey("should succesfully update existing templates", func() {
			So(err, ShouldBeNil)

			newPlaintext := "new secret"
			secret := &manager.Secret{
				Name:           "systems/sandstorm-agent/development/test_secret",
				Plaintext:      []byte(newPlaintext),
				DoNotBroadcast: false,
			}

			err = agent.manager.Put(secret)
			So(err, ShouldBeNil)

			agent.Dwell = 6
			err = agent.runOnUpdate()
			So(err, ShouldBeNil)

			for _, template := range agent.State.Templates {
				secret, exists := template.Secrets["systems/sandstorm-agent/development/test_secret"]
				if exists {
					So(string(secret.Plaintext), ShouldEqual, newPlaintext)
					So(secret.UpdatedAt, ShouldEqual, secret.UpdatedAt)
				}

			}

			testPostRunState(agent, testFolderPath, false)
		})
	})
}

func TestAgentHUPSignalHandler(t *testing.T) {
	Convey("->HUPSignalHandler", t, func() {
		testFolderPath := prepareTestPath("agent-HUPSignalHandler-test")
		agent := getTestAgent(testFolderPath, true, false)
		err := agent.Start(false, true)
		So(err, ShouldBeNil)

		// remove templated out configs to test signal handler
		if fileExists(testFolderPath) {
			for _, template := range agent.State.Templates {
				err := os.Remove(template.Destination)
				So(err, ShouldBeNil)
			}
			err := os.Remove(path.Join(testFolderPath, "touchMe"))
			So(err, ShouldBeNil)

		}
		newAgent := getTestAgent(testFolderPath, true, false)

		for _, template := range agent.State.Templates {
			So(fileExists(template.Destination), ShouldBeFalse)
		}

		So(fileExists(path.Join(testFolderPath, "touchMe")), ShouldBeFalse)
		Convey("->rebaseTemplates", func() {
			replacementTemplates := []*Template{}
			source := path.Join(getTestConfigPath(), "/templates.d/sample_template")
			destination := path.Join(testFolderPath, "template_test")
			command := "ls -la"
			template := NewTemplate(source, destination, command)
			Convey("should copy over secret from same template", func() {
				secret := &manager.Secret{
					Name: "systems/sandstorm-agent/development/test_secret",
				}
				template.Secrets[secret.Name] = secret
				replacementTemplates = append(replacementTemplates, template)

				newSecretsArr, err := agent.rebaseTemplates(replacementTemplates)
				So(err, ShouldBeNil)

				So(len(newSecretsArr), ShouldEqual, 0)

				So(string(replacementTemplates[0].Secrets[secret.Name].Plaintext), ShouldNotEqual, "")
			})

			Convey("Should recognize new secret", func() {
				secret := &manager.Secret{
					Name: "systems/sandstorm-agent/development/test_secret2",
				}
				template.Secrets[secret.Name] = secret
				replacementTemplates = append(replacementTemplates, template)

				newSecretsArr, err := agent.rebaseTemplates(replacementTemplates)
				So(err, ShouldBeNil)

				So(newSecretsArr[0], ShouldEqual, secret.Name)

			})

		})

		Convey("should successfully retemplate out configs and run restart command", func() {
			newSecrets, err := agent.rebaseTemplates(newAgent.State.Templates)
			So(err, ShouldBeNil)
			*agent = *newAgent
			err = agent.sync(newSecrets, false, false, 0)
			So(err, ShouldBeNil)

			for _, template := range agent.State.Templates {
				So(template.Error, ShouldBeEmpty)
				So(fileExists(template.Destination), ShouldBeTrue)
				for _, secret := range template.Secrets {
					So(string(secret.Plaintext), ShouldNotEqual, "")
				}
			}

			So(fileExists(path.Join(testFolderPath, "touchMe")), ShouldBeTrue)
		})

	})
}

func TestAgentFetchSecret(t *testing.T) {
	Convey("->FetchSecret", t, func() {
		testFolderPath := prepareTestPath("agent-fetchSecret-test")
		agent := getTestAgent(testFolderPath, true, false)
		agent.MaxWaitTime = 1
		Convey("should fail and not retry", func() {
			const secretName = "systems/sandstorm-agent/development/backoff_test_secret"
			mockManager := new(mgrMocks.API)
			mockManager.On("Get", secretName).Return(nil, nil)
			agent.manager = mockManager
			mockBackoff := new(mocks.BackOff)
			mockBackoff.On("Reset").Return()
			mockBackoff.On("NextBackOff").Return(backoff.Stop)
			agent.newBackoff = func() backoff.BackOff { return mockBackoff }
			secret, err := agent.FetchSecret(secretName)
			So(secret, ShouldBeNil)
			So(err, ShouldNotBeNil)
			So(mockBackoff.AssertNotCalled(t, "NextBackOff"), ShouldBeTrue)
			So(agent.State.BlockedSecrets, ShouldContainKey, secretName)
		})

		Convey("should fail after trying for testMaxWaitTime seconds", func() {
			mockManager := new(mgrMocks.API)
			mockManager.On("Get", mock.Anything).Return(nil, errors.New("AWS THROTTLE ERROR"))
			agent.manager = mockManager
			mockBackoff := new(mocks.BackOff)
			mockBackoff.On("Reset").Return()
			mockBackoff.On("NextBackOff").Return(backoff.Stop)
			agent.newBackoff = func() backoff.BackOff { return mockBackoff }
			secret, err := agent.FetchSecret("systems/sandstorm-agent/development/backoff_test_secret")
			So(err, ShouldNotBeNil)
			So(secret, ShouldBeNil)
			So(mockBackoff.AssertNumberOfCalls(t, "NextBackOff", 1), ShouldBeTrue)
		})

		Convey("should succeed ", func(c C) {
			secret := &manager.Secret{
				Name:           "systems/sandstorm-agent/development/backoff_test_secret",
				Plaintext:      []byte("backoff secret test"),
				DoNotBroadcast: false,
			}
			mockManager := new(mgrMocks.API)
			mockManager.On("Put", secret).Return(nil, nil)
			mockManager.On("Get", secret.Name).Return(secret, nil)

			agent.manager = mockManager

			mockBackoff := new(mocks.BackOff)
			mockBackoff.On("Reset").Return()
			mockBackoff.On("NextBackOff").Return(backoff.Stop)
			err := agent.manager.Put(secret)
			So(err, ShouldBeNil)
			secret, err = agent.FetchSecret("systems/sandstorm-agent/development/backoff_test_secret")
			So(err, ShouldBeNil)
			So(secret, ShouldNotBeNil)
			So(mockBackoff.AssertNotCalled(t, "NextBackOff"), ShouldBeTrue)
		})

	})
}

func checkDestinationTemplates(agent *Agent) {

	for _, template := range agent.State.Templates {
		So(fileExists(template.Destination), ShouldBeTrue)
		bytes, err := ioutil.ReadFile(template.Destination)
		So(err, ShouldBeNil)

		secrets := template.Secrets
		for name := range secrets {
			secret, err := agent.manager.Get(name)
			So(err, ShouldBeNil)
			So(string(bytes), ShouldContainSubstring, string(secret.Plaintext))
		}
	}
}

func testPostRunStateFile(testFolderPath string) {
	// not an easy way to use the testFolderPath here unforutunately
	stateFilePath := path.Join(getTestConfigPath(), StateFileName)
	So(fileExists(stateFilePath), ShouldBeTrue)

	info, err := os.Stat(stateFilePath)
	So(err, ShouldBeNil)
	So(info.Mode(), ShouldEqual, 0660)
}

func testPostRunState(agent *Agent, testFolderPath string, restartDisabled bool) {

	checkDestinationTemplates(agent)
	testPostRunStateFile(agent.configFolderPath)

	if !restartDisabled {
		// test restart commands
		So(fileExists(path.Join(testFolderPath, "touchMe")), ShouldBeTrue)
	}
}
