package agent

import (
	"errors"
	"net/http"
	"path"
	"strconv"
	"testing"

	"github.com/aws/aws-sdk-go/aws"
	. "github.com/smartystreets/goconvey/convey"
)



func TestConfig(t *testing.T) {

	// Only pass t into top-level Convey calls
	Convey("For A Given Configuration", t, func() {

		testFolderPath := prepareTestPath("test-config")

		Convey("->agentFromConfig()", func() {

			Convey("Should error if given an non-absolute folder path", func() {
				agent, err := agentFromConfig("./test")
				So(err, ShouldNotBeNil)
				So(agent, ShouldBeNil)
			})

			Convey("Should successfully build an agent for a valid path", func() {
				agent, err := agentFromConfig(getTestConfigPath())
				So(err, ShouldBeNil)
				So(agent, ShouldNotBeNil)

				// Ghetto way to check things, but with so many fields, better
				// safe than sorry
				So(agent.Dwell, ShouldEqual, 20)
				So(agent.Splay, ShouldEqual, 5)
				So(agent.TopicARN, ShouldEqual, "arn:aws:sns:us-west-2:734326455073:sandstorm-testing")
				So(agent.KeyID, ShouldEqual, "alias/sandstorm-testing")
				So(agent.TableName, ShouldEqual, "sandstorm-testing")
				So(agent.RoleARN, ShouldEqual, testRoleArn)
				So(agent.Region, ShouldEqual, "us-west-2")
				So(agent.Logging, ShouldEqual, false)
				So(agent.Port, ShouldEqual, 1337)
			})
		})

		agent, err := agentFromConfig(getTestConfigPath())
		So(err, ShouldBeNil)
		So(agent, ShouldNotBeNil)

		Convey("->loadTemplates()", func() {
			templates, err := loadTemplates(agent.configFolderPath)
			So(err, ShouldBeNil)
			So(len(templates), ShouldEqual, 3)

			for _, template := range templates {
				So(template.Source, ShouldNotBeEmpty)
				So(template.Destination, ShouldNotBeEmpty)
				So(template.Content, ShouldNotBeEmpty)
				So(template.Dirty, ShouldBeTrue)
				So(template.FileMode, ShouldNotBeEmpty)
				So(template.UserID, ShouldNotBeEmpty)
				So(template.GroupID, ShouldNotBeEmpty)
			}
		})

		Convey("->syncWithStateFile()", func() {

			templates, err := loadTemplates(agent.configFolderPath)
			So(err, ShouldBeNil)
			So(templates, ShouldNotBeEmpty)
			for _, template := range templates {
				template.FileComparer = mockAlwaysTrueFileComparer
			}

			stateFilePath := path.Join(testFolderPath, StateFileName)

			Convey("should not error if no state file is present", func() {
				err := syncWithStateFile(stateFilePath, templates)
				So(err, ShouldBeNil)
				So(dirtyTemplateCount(templates), ShouldEqual, 3)
			})

			Convey("should mark up-to-date templates from a statefile as clean", func() {
				cleanTemplates := []*Template{templates[0]}
				fileState := newStateFile(cleanTemplates)
				err := fileState.ToFile(stateFilePath)
				So(err, ShouldBeNil)

				err = syncWithStateFile(stateFilePath, templates)
				So(err, ShouldBeNil)
				So(dirtyTemplateCount(templates), ShouldEqual, 2)
				So(templates[0].Dirty, ShouldBeFalse)
			})

			Convey("should not clean templates with mismatched content", func() {
				copy := copyTemplate(templates[0])
				copy.Content = "foobarred"
				cleanTemplates := []*Template{copy}
				fileState := newStateFile(cleanTemplates)
				err := fileState.ToFile(stateFilePath)
				So(err, ShouldBeNil)

				err = syncWithStateFile(stateFilePath, templates)
				So(err, ShouldBeNil)
				So(dirtyTemplateCount(templates), ShouldEqual, 3)
				So(templates[0].Dirty, ShouldBeTrue)
			})
		})

		Convey("->loadAgent()", func() {

			cleanAgentFiles(getTestConfigPath())

			Convey("should populate templates and secrets", func() {
				agent, err := loadAgent(getTestConfigPath(), true)
				So(err, ShouldBeNil)

				So(agent.State.Templates, ShouldNotBeEmpty)
				So(len(agent.State.Templates), ShouldEqual, 3)
				So(dirtyTemplateCount(agent.State.Templates), ShouldEqual, 3)
			})
		})

		Convey("->configureAWS()", func() {
			Convey("good configuration should succeed", func() {
				agent, err := loadAgent(getTestConfigPath(), true)
				So(err, ShouldBeNil)
				awsConfig := agent.createAWSConfig()
				agent.awsConfig = awsConfig
				agent.Queue, agent.manager, err = agent.configureAWS(awsConfig)
				So(err, ShouldBeNil)
				So(agent.awsConfig, ShouldNotBeNil)
				So(agent.Queue, ShouldNotBeNil)
				So(agent.manager, ShouldNotBeNil)
			})

			Convey("should provide an error that is not isSystemError if improper account", func() {
				agent, err := loadAgent(getTestConfigPath(), true)
				So(err, ShouldBeNil)
				awsConfig := agent.createAWSConfig()
				agent.awsConfig = awsConfig
				agent.RoleARN = "default"
				agent.sandstormAccountNumber = "test_account_number"
				agent.Queue, agent.manager, err = agent.configureAWS(awsConfig)
				So(err, ShouldNotBeEmpty)
				So(err, ShouldHaveSameTypeAs, &IncorrectAWSAccount{})
				So(isSystemError(err), ShouldBeFalse)
			})

			Convey("should provide an error that is not isSystemError if access denied to resource", func() {
				agent, err := loadAgent(getTestConfigPath(), true)
				So(err, ShouldBeNil)
				awsConfig := agent.createAWSConfig()
				agent.awsConfig = awsConfig
				agent.RoleARN = "dderpderpderpderpderpderpderpderpderpderpderperp"
				agent.Queue, agent.manager, err = agent.configureAWS(awsConfig)
				So(isSystemError(err), ShouldBeFalse)
			})

		})

		Convey("Proxy Tests", func() {
			agent, err := loadAgent(getTestConfigPath(), true)
			So(err, ShouldBeNil)
			awsConfigFail := &aws.Config{Region: aws.String(agent.Region)}
			DefaultTransportFail := &http.Transport{}

			rt := newMockRoundTripper(DefaultTransportFail, errors.New("can't reach sqs"))
			defaultHTTPClient := &http.Client{Transport: rt}
			awsConfigFail.HTTPClient = defaultHTTPClient

			agent.DefaultQueue, agent.defaultManager, err = agent.configureAWS(awsConfigFail)
			So(err, ShouldNotBeNil)
			agent.Queue = agent.DefaultQueue
			agent.manager = agent.defaultManager

			awsConfig := agent.createAWSConfig()
			agent.awsConfig = awsConfig
			agent.ProxyQueue, agent.proxyManager, err = agent.configureAWS(awsConfig)
			So(err, ShouldBeNil)

		})

		Convey("->Build()", func() {
			agent, err := Build(getTestConfigPath(), false, true)
			So(err, ShouldBeNil)
			So(agent, ShouldNotBeNil)
		})

		Convey("->loadSimulatorValues()", func() {
			simlateValues, err := loadSimulatorValues(getTestConfigPath())
			So(err, ShouldBeNil)
			So(simlateValues["systems/sandstorm-agent/development/test_secret"].(string), ShouldEqual, testDummyString)
			So(simlateValues["systems/sandstorm-agent/development/test_secret_2"].(float64), ShouldEqual, testDummyInt)

		})

		Convey("->BuildSimulatorAgent()", func() {
			agent, err := BuildSimulatorAgent(getTestConfigPath())
			So(err, ShouldBeNil)
			So(agent, ShouldNotBeNil)

			for _, template := range agent.State.Templates {
				for _, secret := range template.Secrets {
					if secret.Name == "systems/sandstorm-agent/development/test_secret" {
						So(string(secret.Plaintext), ShouldEqual, testDummyString)
					}

					if secret.Name == "systems/sandstorm-agent/development/test_secret_2" {
						So(string(secret.Plaintext), ShouldEqual, strconv.Itoa(testDummyInt))
					}
				}
			}
		})
		Convey("->CreatePIDFile()", func() {
			err := CreatePIDFile(testPIDPath)
			So(err, ShouldBeNil)
			So(fileExists(testPIDPath), ShouldBeTrue)
		})
	})
}

func dirtyTemplateCount(templates []*Template) int {
	dirty := 0
	for _, template := range templates {
		if template.Dirty {
			dirty++
		}
	}
	return dirty
}

type mockRoundTripper struct {
	http.RoundTripper
	pop string
	err error
}

func (mrt mockRoundTripper) RoundTrip(Request *http.Request) (*http.Response, error) {
	if mrt.err != nil {
		return nil, mrt.err
	}
	return nil, nil
}

func newMockRoundTripper(rt http.RoundTripper, err error) http.RoundTripper {
	return &mockRoundTripper{
		RoundTripper: rt,
		err:          err,
	}
}
