// +build integration

package agent

import (
	"encoding/json"
	"log"
	"net/http"
	"net/http/httptest"
	"net/url"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
)

func TestEndpoints(t *testing.T) {
	// Setup our basic server stack, and a reachable url endpoint for it
	testFolderPath := prepareTestPath("test-endpoints")
	agent := getTestAgent(testFolderPath, true, false)
	router := agent.BuildRouter()
	if router == nil {
		log.Fatal("could not build router")
	}
	server := httptest.NewServer(router)
	baseURL := server.URL

	Convey("Endpoint Tests", t, func() {

		Convey("->buildHealthProfile()", func() {

			Convey("should successfully build a profile from it's agent", func() {
				profile, err := agent.buildHealthProfile()
				So(profile, ShouldNotBeNil)
				So(err, ShouldBeNil)
			})
		})

		Convey("->BuildRouter()", func() {

			Convey("/health", func() {

				err := agent.runOnUpdate()
				So(err, ShouldBeNil)

				Convey("should return a formatted output page", func() {
					healthURL, err := router.Get("healthcheck").URL()
					So(err, ShouldBeNil)
					healthURL, err = combineURL(baseURL, healthURL)
					So(err, ShouldBeNil)

					resp, err := http.Get(healthURL.String())
					So(err, ShouldBeNil)
					So(resp, ShouldNotBeNil)
					So(resp.StatusCode, ShouldEqual, 200)

					// Now compare our local health profile vs the returned one
					var profileResponse *HealthProfile
					decoder := json.NewDecoder(resp.Body)
					err = decoder.Decode(&profileResponse)
					So(err, ShouldBeNil)

					err = resp.Body.Close()
					So(err, ShouldBeNil)
				})
			})

			Convey("/version", func() {
				versionURL, err := router.Get("version").URL()
				So(err, ShouldBeNil)

				versionURL, err = combineURL(baseURL, versionURL)
				So(err, ShouldBeNil)

				resp, err := http.Get(versionURL.String())
				So(err, ShouldBeNil)
				So(resp, ShouldNotBeNil)
				So(resp.StatusCode, ShouldEqual, 200)

				err = resp.Body.Close()
				So(err, ShouldBeNil)
			})
		})
	})
	server.Close()
}

func combineURL(baseURL string, routeURL *url.URL) (*url.URL, error) {

	base, err := url.Parse(baseURL)
	if err != nil {
		return nil, err
	}

	routeURL.Scheme = base.Scheme
	routeURL.Host = base.Host
	return routeURL, nil
}
