"use strict";

var expect = require('chai').expect;
var LambdaTester = require('lambda-tester');
var myLambda = require('../DynamoLambda');
var AWS = require('aws-sdk-mock');

var shouldPublishToSNSMap = {
  "if event INSERT and do_not_broadcast: false.": buildDynmoDbImage(
    'INSERT', false, true, false, true),
  "if event INSERT and do_not_broadcast: false other PARMS: N/A.": buildDynmoDbImage(
    'INSERT', false, true, true, true),
  "if event INSERT and do_not_broadcast: false others N/A.": buildDynmoDbImage(
    'INSERT', false, false, false, false),
  "if event INSERT and do_not_broadcast: false other params are N/A": buildDynmoDbImage(
    'INSERT', false, false, true, false),
  "if event MODIFY and do_not_broadcast: false and secret have changed. .": buildDynmoDbImage(
    'MODIFY', false, true, false, true),
  "if event MODIFY and do_not_broadcast: false and old broadcast was true but secrets did not change.": buildDynmoDbImage(
    'MODIFY', false, false, true, false),
  "if event MODIFY and do_not_broadcast: false and old broadcast was true and secret changed.": buildDynmoDbImage(
    'MODIFY', false, true, true, true),
  "if event MODIFY and secret did not change but version changed.": buildDynmoDbImage(
    'MODIFY', false, false, false, true),
};

var shouldNotBePublishedToSNSMap = {
  "if event is not MODIFY or INSERT.": buildDynmoDbImage('DELETE',
    false, true, false, true),
  'if do_not_broadcast is set to true.': buildDynmoDbImage('MODIFY',
    true, true, false, true),
  'event is MODIFY and secret and version did not change': buildDynmoDbImage(
    'MODIFY', false, false, false, false),
};

function buildDynmoDbImage(eventName, doNotBroadcast, secretChanged,
  oldDoNotBroadcast, versionChanged) {
  var name = "syseng/testerino/production/bachtest";

  var image = {
    "do_not_broadcast": {
      "BOOL": doNotBroadcast
    },
    "updated_at": {
      "N": "1479410190"
    },
    "name": {
      "S": name
    },
    "value": {
      "S": "oH12GQxeqEIbKt4x6SF2PybtTd8tbOhYxCFFZAw7G2Y="
    },
    "key": {
      "S": "AQEDAHgy/f1oX0usuW2CHF9qAhL8A41L+LMuqj9kliVP4DdGkgAAAH4wfAYJKoZIhvcNAQcGoG8wbQIBADBoBgkqhkiG9w0BBwEwHgYJYIZIAWUDBAEuMBEEDDOG9Ric8jmVUVNWxwIBEIA7sxH7ZOtbBL2S3yb2AAGRBhbqmhrveejCnjMe55QmXnLFt9+bri+4Gp0bSCLxFZnNMyXbx5bgFKVniOw="
    }
  };

  var json = {
    "eventName": eventName,
    "dynamodb": {
      "Keys": {
        "name": {
          "S": name
        }
      },
      "NewImage": image,
    },
    "eventSourceARN": "arn:aws:dynamodb:us-west-2:734326455073:table/sandstorm-testing/stream/2016-02-04T00:53:06.459"
  };

  if (eventName == 'MODIFY') {
    var oldImage = JSON.parse(JSON.stringify(image));
    if (secretChanged === true) {
      oldImage.value.S = 'oH12GQxeqEIbKt4x6SF2PybtTd8tbOhYxCFFZAw7F2Y=';
    }
    if (versionChanged === true) {
      oldImage.updated_at.N = 1179480100;
    }

    oldImage.do_not_broadcast.BOOL = oldDoNotBroadcast;
    json.dynamodb.OldImage = oldImage;
  }
  return json;
}

describe('Sandstorm lambda should publish to SNS', function() {
  beforeEach(function(done) {
    AWS.mock('SNS', 'publish', function(params, callback) {
      callback(null, 'Success!');
    });
    done();
  });

  afterEach(function(done) {
    AWS.restore('SNS', 'publish');
    done();
  });

  Object.keys(shouldPublishToSNSMap).forEach(function(message) {
    var json = shouldPublishToSNSMap[message];
    it(message, function(done) {

      LambdaTester(myLambda.handler)
        .event({
          Records: [json]
        })
        .expectSucceed(function(result) {
          expect(result.processedEvent).to.be.true;
        })
        .verify(done);
    });
  });

});

describe('Sandstorm lambda fails to publish if SNS fails.', function() {
  beforeEach(function(done) {
    AWS.mock('SNS', 'publish', function(params, callback) {
      callback('Failure!');
    });
    done();
  });

  afterEach(function(done) {
    AWS.restore('SNS', 'publish');
    done();
  });

  Object.keys(shouldPublishToSNSMap).forEach(function(message) {
    var json = shouldPublishToSNSMap[message];
    it(message, function(done) {

      LambdaTester(myLambda.handler)
        .event({
          Records: [json]
        })
        .expectFail(function(result) {
          expect(result.processedEvent).to.be.false;
        })
        .verify(done);
    });
  });

  Object.keys(shouldNotBePublishedToSNSMap).forEach(function(message) {
    var json = shouldNotBePublishedToSNSMap[message];
    it(message, function(done) {
      LambdaTester(myLambda.handler)
        .event({
          Records: [json]
        })
        .expectSucceed(function(result) {
          expect(result.processedEvent).to.be.true;
        })
        .verify(done);
    });
  });
});
