package agent

import (
	"bytes"
	"os"
	"path"
	"testing"

	"code.justin.tv/systems/sandstorm/manager"

	. "github.com/smartystreets/goconvey/convey"
)

func TestTemplate(t *testing.T) {

	Convey("Template Tests", t, func() {

		testFolderPath := prepareTestPath("test-template")

		Convey("->execCommand()", func() {
			out, stderr, exitCode, err := execCommand("ls -la")
			So(out, ShouldNotBeNil)
			So(stderr, ShouldEqual, "")
			So(exitCode, ShouldEqual, 0)
			So(err, ShouldBeNil)
		})

		source := path.Join(getTestConfigPath(), "/templates.d/sample_template")
		destination := path.Join(testFolderPath, "template_test")
		command := "ls -la"

		template := NewTemplate(source, destination, command)

		Convey("->Initialize()", func() {

			Convey("should load content and set affected values", func() {

				err := template.Initialize()
				So(err, ShouldBeNil)

				So(template.CommandStatus.Status, ShouldEqual, statusStartup)
				So(template.Content, ShouldNotBeEmpty)
				So(len(template.Secrets), ShouldEqual, 1)

				_, exists := template.Secrets["systems/sandstorm-agent/development/test_secret"]
				So(exists, ShouldBeTrue)
			})
		})

		Convey("CompareFileStat", func() {
			Convey("with same file owner and filemode", func() {
				tmpfile, old, err := newTemplateFileTest()
				So(err, ShouldBeNil)
				So(CompareFileStat(old, old), ShouldBeTrue)

				So(os.Remove(tmpfile.Name()), ShouldBeNil)
				So(tmpfile.Close(), ShouldBeNil)
			})

			Convey("with mismatch file owner", func() {
				tmpfile, old, err := newTemplateFileTest()
				So(err, ShouldBeNil)
				new := new(Template)
				*new = *old
				new.UserID = old.UserID + 1
				So(CompareFileStat(new, old), ShouldBeFalse)

				So(os.Remove(tmpfile.Name()), ShouldBeNil)
				So(tmpfile.Close(), ShouldBeNil)
			})

			Convey("with mismatch gid", func() {
				tmpfile, old, err := newTemplateFileTest()
				So(err, ShouldBeNil)
				new := new(Template)
				*new = *old
				new.GroupID = old.GroupID + 1
				So(CompareFileStat(new, old), ShouldBeFalse)

				So(os.Remove(tmpfile.Name()), ShouldBeNil)
				So(tmpfile.Close(), ShouldBeNil)
			})

			Convey("with mismatch file mode", func() {
				tmpfile, old, err := newTemplateFileTest()
				So(err, ShouldBeNil)

				new := new(Template)
				*new = *old
				new.FileMode = os.FileMode(0666)
				So(CompareFileStat(new, old), ShouldBeFalse)

				So(os.Remove(tmpfile.Name()), ShouldBeNil)
				So(tmpfile.Close(), ShouldBeNil)
			})
		})

		Convey("For a given Template", func() {

			Convey("->AttemptCommand()", func() {

				Convey("should successfully run for a valid restart command", func() {
					err := template.AttemptCommand(1)
					So(err, ShouldBeNil)
					So(template.CommandStatus.Status, ShouldEqual, statusSuccess)
				})

				Convey("should error gracefully for an invalid restart command", func() {
					template.Command = "foo bar"
					err := template.AttemptCommand(1)
					So(err, ShouldNotBeNil)
					So(template.CommandStatus.Status, ShouldEqual, statusFail)
				})
			})
		})

		Convey("->copyTemplate()", func() {
			err := template.Initialize()
			So(err, ShouldBeNil)

			Convey("should deep copy successfully", func() {
				copy := copyTemplate(template)

				copy.Source = "new_source"
				So(template.Source, ShouldEqual, source)

				for key, secret := range copy.Secrets {
					newPlainText := []byte("asdasnbjdsajk")
					secret.Plaintext = newPlainText
					So(template.Secrets[key].Plaintext, ShouldNotEqual, newPlainText)
					break
				}
			})
		})

		Convey("->upsertSecret()", func() {

			// set this to false, so we have a control to work against
			template.Dirty = false

			testSecret := &manager.Secret{
				Name:      "testSecret",
				Plaintext: []byte{},
				UpdatedAt: 0,
			}

			Convey("should add initialization template secret", func() {
				template.UpsertSecret(testSecret)

				So(len(template.Secrets), ShouldEqual, 1)
				So(template.Dirty, ShouldBeFalse)
			})

			Convey("should update initialization template secret", func() {
				template.UpsertSecret(testSecret)

				updatedSecret := &manager.Secret{
					Name:      "testSecret",
					Plaintext: []byte("Notempty"),
					UpdatedAt: 0,
				}

				template.UpsertSecret(updatedSecret)

				So(len(template.Secrets), ShouldEqual, 1)
				So(template.Secrets[testSecret.Name], ShouldEqual, updatedSecret)
				So(template.Dirty, ShouldBeFalse)
			})

			Convey("should update older secret", func() {
				testSecret.Plaintext = []byte("notempty")
				testSecret.UpdatedAt = 1
				template.UpsertSecret(testSecret)

				updatedSecret := &manager.Secret{
					Name:      "testSecret",
					Plaintext: []byte("Notempty"),
					UpdatedAt: 2,
				}

				template.UpsertSecret(updatedSecret)

				So(len(template.Secrets), ShouldEqual, 1)
				So(template.Secrets[testSecret.Name], ShouldEqual, updatedSecret)
				So(template.Dirty, ShouldBeTrue)
			})

			Convey("should do nothing for same secret version", func() {
				testSecret.Plaintext = []byte("notempty")
				testSecret.UpdatedAt = 1
				template.UpsertSecret(testSecret)

				updatedSecret := &manager.Secret{}
				*updatedSecret = *testSecret

				template.UpsertSecret(updatedSecret)

				So(len(template.Secrets), ShouldEqual, 1)
				So(template.Secrets[testSecret.Name], ShouldEqual, testSecret)
				So(template.Dirty, ShouldBeFalse)
			})

			Convey("should update a reverted secret", func() {
				testSecret.UpdatedAt = 15
				testSecret.Plaintext = []byte("secretInTheFuture")
				template.UpsertSecret(testSecret)

				updatedSecret := &manager.Secret{}
				*updatedSecret = *testSecret
				updatedSecret.Plaintext = []byte("previous secret")
				updatedSecret.UpdatedAt = 1

				template.UpsertSecret(updatedSecret)

				So(len(template.Secrets), ShouldEqual, 1)
				So(template.Secrets[testSecret.Name], ShouldEqual, updatedSecret)
				So(template.Dirty, ShouldBeTrue)
			})
		})
		Convey("->synchronizeSecrets", func() {
			newPlaintext := "new secret"
			secret := &manager.Secret{
				Name:           "systems/sandstorm-agent/development/test_secret",
				Plaintext:      []byte(newPlaintext),
				DoNotBroadcast: false,
			}
			replacementTemplate := NewTemplate(source, destination, command)
			newSecretsSet := map[string]bool{}

			Convey("should copy over same secret", func() {

				template.Secrets[secret.Name] = secret
				replacementSecret := &manager.Secret{
					Name: "systems/sandstorm-agent/development/test_secret",
				}
				replacementTemplate.Secrets[replacementSecret.Name] = replacementSecret
				replacementTemplate.synchronizeSecrets(template, newSecretsSet)
				So(newSecretsSet, ShouldBeEmpty)
				So(string(replacementTemplate.Secrets[replacementSecret.Name].Plaintext), ShouldEqual, newPlaintext)
			})
			Convey("should recognize new secret", func() {
				replacementSecret := &manager.Secret{
					Name: "systems/sandstorm-agent/development/test_secret2",
				}
				replacementTemplate.Secrets[replacementSecret.Name] = replacementSecret
				replacementTemplate.synchronizeSecrets(template, newSecretsSet)
				So(newSecretsSet, ShouldContainKey, replacementSecret.Name)
				So(replacementTemplate.Secrets, ShouldNotContainKey, secret.Name)
			})
		})

		Convey("->Output()", func() {

			Convey("should be exactly len(secret.Plaintext) bytes in output file", func() {
				newPlaintext := "beep"
				secret := &manager.Secret{
					Name:           "systems/sandstorm-agent/development/test_secret",
					Plaintext:      []byte(newPlaintext),
					DoNotBroadcast: false,
				}
				template := NewTemplate(source, destination, command)
				template.Secrets["systems/sandstorm-agent/development/test_secret"] = secret
				template.Content = "{{ key \"systems/sandstorm-agent/development/test_secret\"}}\n{{- /*suppress trailing new line*/ -}}"

				err := template.Output()
				So(err, ShouldBeNil)

				file, err := os.Open(destination)
				So(err, ShouldBeNil)
				fi, err := file.Stat()
				So(err, ShouldBeNil)

				data := make([]byte, fi.Size())

				_, err = file.Read(data)
				So(err, ShouldBeNil)

				So(bytes.Compare(data, secret.Plaintext), ShouldEqual, 0)
				So(fi.Size(), ShouldEqual, len(secret.Plaintext))
			})

		})
	})
}
