// +build integration

package apiserver

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"net/http/httptest"
	"strings"
	"testing"
	"time"

	"golang.org/x/oauth2"

	"code.justin.tv/systems/guardian/guardian"
	apiserverMocks "code.justin.tv/systems/sandstorm/apiserver/mocks"
	"code.justin.tv/systems/sandstorm/manager"
	managerMocks "code.justin.tv/systems/sandstorm/manager/mocks"
	globalMocks "code.justin.tv/systems/sandstorm/mocks"
	"code.justin.tv/systems/sandstorm/testutil"
	"code.justin.tv/systems/sandstorm/util"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/sts"
	"github.com/urfave/negroni"
	"github.com/satori/go.uuid"
	. "github.com/smartystreets/goconvey/convey"
)

const (
	//Test Access token from guardian (staging) for Ladislav Kovacs user. MUST HAVE ACCESS TO "team-syseng"
	TestAccessToken = "62d661843680d407b6127d9c2e705eceab59f15e"
	//Test Access token from guardian (staging) for Guardian TestUser user. MUST NOT HAVE ACCESS TO "team-syseng"
	LimitedAccessToken = "b99843d903dd807f5b676cec5f28c771e8ffc470"
)

func createTestManager() *manager.Manager {
	config, err := testutil.LoadTestConfig()
	So(err, ShouldBeNil)
	So(config, ShouldNotBeNil)
	awsConfig := &aws.Config{
		Region: aws.String(config.Sandstorm.Region),
	}
	stsclient := sts.New(session.New(awsConfig))
	arp := &stscreds.AssumeRoleProvider{
		Duration:     900 * time.Second,
		ExpiryWindow: 10 * time.Second,
		RoleARN:      config.Sandstorm.RoleArn,
		Client:       stsclient,
	}
	awsConfig.WithCredentials(credentials.NewCredentials(arp))
	return manager.New(manager.Config{
		AWSConfig: awsConfig,
		TableName: config.Sandstorm.TableName,
		KeyID:     config.Sandstorm.KeyID,
	})
}

type mockedServices struct {
	mgr        *managerMocks.API
	authorizer *apiserverMocks.OAuthAuthorizer
	ddb        *globalMocks.DynamoDBAPI
	iam        *globalMocks.IAMAPI
	logger     *globalMocks.FieldLogger

	// map of dynamo table names to table arns
	tableArns map[string]string
}

func setupMockAPI(t *testing.T) (server *httptest.Server, mocked *mockedServices) {
	cfg, err := LoadConfig("../test.hcl")
	if err != nil {
		t.Error(err)
	}

	mocked = &mockedServices{
		mgr:        new(managerMocks.API),
		authorizer: new(apiserverMocks.OAuthAuthorizer),
		ddb:        new(globalMocks.DynamoDBAPI),
		iam:        new(globalMocks.IAMAPI),
		tableArns:  make(map[string]string),
	}

	svc := newService(cfg, &clients{
		changelog:       new(changelogClient),
		mgr:             mocked.mgr,
		oauthAuthorizer: mocked.authorizer,
		ddb:             mocked.ddb,
		iam:             mocked.iam,
		logger:          util.SetupLogging(cfg.Syslog),
	})

	mockOAuthMiddleware := negroni.HandlerFunc(func(w http.ResponseWriter, r *http.Request, next http.HandlerFunc) {
		testUser := &guardian.User{
			CN:  "Ladislav Kovács",
			UID: "GuardiaN",
			Groups: []string{
				"team-syseng",
				"team-navi",
			},
		}
		next(w, setGuardianUser(r, testUser))
	})

	svc.oauthMiddleware = mockOAuthMiddleware

	sandstormTable := cfg.Sandstorm.TableName
	auditTable := cfg.Sandstorm.TableName + "_audit"
	namespaceTable := cfg.Sandstorm.TableName + "_namespace"

	mocked.mgr.On("TableName").Return(sandstormTable).Once()
	mocked.mgr.On("AuditTableName").Return(auditTable).Once()
	mocked.mgr.On("NamespaceTableName").Return(namespaceTable).Once()

	for _, v := range []string{sandstormTable, auditTable, namespaceTable} {
		mocked.tableArns[v] = "arn:aws:dynamodb:us-west-2:123:table/" + v
		mocked.ddb.On("DescribeTable", &dynamodb.DescribeTableInput{TableName: aws.String(v)}).Return(&dynamodb.DescribeTableOutput{Table: &dynamodb.TableDescription{TableArn: aws.String(mocked.tableArns[v])}}, nil).Once()
	}

	err = svc.setup()
	if err != nil {
		t.Error(err)
	}

	server = httptest.NewServer(svc)
	return
}

func parseBody(body io.ReadCloser) ([]byte, error) {
	jsonBytes, err := ioutil.ReadAll(body)
	defer func() {
		err := body.Close()
		So(err, ShouldBeNil)
	}()

	return jsonBytes, err
}

func secretFromBody(body io.ReadCloser) (*manager.Secret, error) {
	jsonBytes, err := parseBody(body)
	if err != nil {
		return nil, err
	}

	var single SecretSingle
	err = json.Unmarshal(jsonBytes, &single)
	if err != nil {
		return nil, err
	}
	return single.Data.Attributes, nil
}

func elementsFromBody(body io.ReadCloser) ([]SecretElement, error) {
	jsonBytes, err := parseBody(body)
	if err != nil {
		return nil, err
	}

	var secrets SecretArray
	err = json.Unmarshal(jsonBytes, &secrets)
	if err != nil {
		return nil, err
	}
	return secrets.Data, nil
}

func testSecretToJSON(secret *manager.Secret) ([]byte, error) {
	result := SecretSingle{Data: secretToSecretElement(secret)}
	return json.Marshal(&result)
}

func createTestSecret() *manager.Secret {
	return &manager.Secret{
		Name:           fmt.Sprintf("navi/de_dust/development/test_secret-%s", uuid.NewV4().String()),
		Plaintext:      []byte("wootootoot123"),
		DoNotBroadcast: false,
	}
}

func copySecretName() string {
	return fmt.Sprintf("navi/de_dust/development/test_secret-copy-%s", uuid.NewV4().String())
}

func testJSONRequest(baseURL string, routeURI string, method string, json []byte, authorize bool) (*http.Response, error) {

	assembledURL := strings.Join([]string{baseURL, routeURI}, "")

	req, err := http.NewRequest(method, assembledURL, bytes.NewBuffer(json))
	if err != nil {
		return nil, err
	}

	req.Header.Set("Origin", "http://localhost")
	req.Header.Set("Content-Type", contentType)
	if authorize {
		token := &oauth2.Token{AccessToken: TestAccessToken}
		token.SetAuthHeader(req)
	}

	client := &http.Client{}
	return client.Do(req)
}

func getSecretFromDynamoDB(mgr *manager.Manager, name string) (map[string]*dynamodb.AttributeValue, error) {
	return getFromDynamoDB(mgr, getSandstormSecretQueryInput(name, mgr.TableName()))
}

func getSecretAuditFromDynamoDB(mgr *manager.Manager, name string, updatedAt int64) (map[string]*dynamodb.AttributeValue, error) {
	return getFromDynamoDB(mgr, getSandstormSecretQueryInput(name, mgr.AuditTableName()))
}

func getFromDynamoDB(mgr *manager.Manager, input *dynamodb.QueryInput) (map[string]*dynamodb.AttributeValue, error) {
	queryOutput, err := mgr.DynamoDB.Query(input)
	if err != nil {
		return nil, err
	}
	return queryOutput.Items[0], nil
}

func getSandstormSecretQueryInput(name string, tableName string) *dynamodb.QueryInput {
	return &dynamodb.QueryInput{
		ConsistentRead: aws.Bool(true),
		ExpressionAttributeNames: map[string]*string{
			"#N": aws.String("name"),
		},
		ExpressionAttributeValues: map[string]*dynamodb.AttributeValue{
			":name": {
				S: aws.String(name),
			},
		},
		ScanIndexForward:       aws.Bool(false),
		KeyConditionExpression: aws.String("#N = :name"),
		ReturnConsumedCapacity: aws.String("INDEXES"),
		TableName:              aws.String(tableName),
	}
}

func getAwsCredentialConfig() *aws.Config {

	awsConfig := &aws.Config{Region: aws.String("us-west-2")}
	sess := session.New(awsConfig)
	stsclient := sts.New(sess)
	arp := &stscreds.AssumeRoleProvider{
		Duration:     900 * time.Second,
		ExpiryWindow: 10 * time.Second,
		RoleARN:      "arn:aws:iam::734326455073:role/sandstorm-apiserver-testing",
		Client:       stsclient,
	}
	credentials := credentials.NewCredentials(arp)
	awsConfig.WithCredentials(credentials)
	return awsConfig
}
