// +build integration

package apiserver

import (
	"bytes"
	"io/ioutil"
	"os"
	"path/filepath"
	"testing"
	"time"

	"code.justin.tv/systems/changelog-clients/go/ccg"

	"github.com/cactus/go-statsd-client/statsd"
	"github.com/cenkalti/backoff"
	"github.com/sirupsen/logrus"
	"github.com/stretchr/testify/assert"
)

func TestChangelogClient(t *testing.T) {
	t.Run("creates valid client", func(t *testing.T) {
		tmpdir, err := ioutil.TempDir("", "changelogClient")
		if err != nil {
			t.Fatal(err)
		}

		defer func() {
			err := os.RemoveAll(tmpdir)
			if err != nil {
				t.Error(err)
			}
		}()

		logDir, err := ioutil.TempDir("", "logDir")
		if err != nil {
			t.Fatal(err)
		}

		defer func() {
			err := os.RemoveAll(logDir)
			if err != nil {
				t.Error(err)
			}
		}()

		cfg := ChangelogConfig{
			Enabled:        true,
			Host:           "changelog-s2s-staging.internal.justin.tv",
			Category:       "sandstorm-apiserver-testing",
			CallerName:     "sandstorm-apiserver-testing",
			LogDir:         logDir,
			BoltDBFilePath: filepath.Join(tmpdir, "test.boltdb"),
		}
		lgr := logrus.New()
		var buf bytes.Buffer
		lgr.Level = logrus.DebugLevel
		lgr.Out = &buf

		client, err := configureChangelog(cfg, lgr, &statsd.NoopClient{})

		if !assert.NoError(t, err) {
			t.Log(buf.String())
			return
		}
		assert.NotNil(t, client.Client)
		assert.Equal(t, lgr, client.logger)

		time.Sleep(5 * time.Second)

		changeEvent := &ccg.Event{
			Category:       "testCategory",
			Criticality:    ccg.SevInfo,
			Target:         "testTarget",
			Description:    "testDescription",
			Username:       "testUsername",
			Command:        "sandstorm-apiserver-testing",
			AdditionalData: "testAdditionalData",
		}

		tries := backoff.NewTicker(&backoff.ConstantBackOff{
			time.Second,
		})

		var maxTries = 30
		for range tries.C {
			err = client.SendEvent(changeEvent)
			if err != nil {
				t.Logf("attempting to SendEvent\nerror: %s", err.Error())
				maxTries--
			} else {
				break
			}

			if maxTries <= 0 {
				break
			}
			continue
		}

		assert.NoError(t, err)
		t.Log(buf.String())
	})
}
