package apiserver

import (
	"errors"
	"net/http"
	"strconv"

	ssErrors "code.justin.tv/systems/sandstorm/errors"
	"github.com/aws/aws-sdk-go/aws/awserr"
)

// APIErrorer provides context needed to return the correct api error response
type APIErrorer interface {
	// Must satisfy base error interface
	error
	// HTTP response code
	Code() int
}

// APIError is a generic implementation of APIErrorer
type APIError struct {
	msg  error
	code int
}

// Error for interface adherence
func (ae APIError) Error() string {
	return ae.msg.Error()
}

// Code represets the http status code of the error
func (ae APIError) Code() int {
	return ae.code
}

// apiError handles API Responses where an error has been raised internally
func (svc *Service) apiError(err error, w http.ResponseWriter, r *http.Request) {
	errorStatus, err := getErrorMessageAndStatusCodeByError(err, r)
	svc.apiErrorStatus(w, r, err, errorStatus)
}

func getErrorMessageAndStatusCodeByError(err error, r *http.Request) (int, error) {
	var errorStatus int
	switch newErr := err.(type) {
	case awserr.RequestFailure:
		// remove struct from error and return string to error
		err = errors.New(newErr.Message())
		errorStatus = newErr.StatusCode()
	case awserr.Error:
		// remove struct from error and return string to error
		err = errors.New(newErr.Message())
		errorStatus = http.StatusInternalServerError
	case APIErrorer:
		errorStatus = newErr.Code()
	default:
		errorStatus = http.StatusInternalServerError
	}
	return errorStatus, err
}

// JSONError is a type representing a JSONAPI error.
type JSONError struct {
	Class  string `json:"class"`
	Detail string `json:"detail"`
	Status string `json:"status"`
	Source struct {
		Pointer string `json:"pointer"`
	} `json:"source"`
}

// JSONErrors represents a list of JSONError objects. Any error response should
// return this.
type JSONErrors struct {
	Errors []JSONError `json:"errors"`
}

// apiErrorStatus returns a JSON formatted error response in the body
// of http.Error for the specified statusCode.
// This also logs the error.
func (svc *Service) apiErrorStatus(w http.ResponseWriter, r *http.Request, responseError error, statusCode int) {
	entry := svc.logEntry(r).WithField("status", statusCode)
	entry.Warnf("error: %s", responseError.Error())

	className := "Error"

	if v, ok := responseError.(ssErrors.ErrorWithClassName); ok {
		className = v.GetClassName()
	}

	jError := JSONError{
		Class:  className,
		Detail: responseError.Error(),
		Status: strconv.Itoa(statusCode),
	}
	jError.Source.Pointer = "data/attributes/name"

	result := JSONErrors{
		Errors: []JSONError{jError},
	}

	err := jsonResponseStatus(result, w, statusCode)
	if err != nil {
		entry := svc.logEntry(r).WithField("status", statusCode)
		entry.Warnf("error encoding JSON error response: %s", err)
	}
}
