// +build integration

package consumedsecrets

import (
	"testing"
	"time"

	"code.justin.tv/systems/sandstorm/inventory/heartbeat"

	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/aws/aws-sdk-go/aws/credentials/stscreds"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/dynamodb"
	"github.com/aws/aws-sdk-go/service/dynamodb/dynamodbiface"
	"github.com/aws/aws-sdk-go/service/sts"
	uuid "github.com/satori/go.uuid"
	"github.com/stretchr/testify/assert"
)

const integrationTestRegion = "us-west-2"
const integrationTestTableName = "heartbeats-testing"
const integretionTestTTL = 86400
const integrationCallerArn = "myCallerArn"
const integrationStormwatchRoleArn = "arn:aws:iam::516651178292:role/inventory-stormwatch-service-testing"
const integrationAdminRoleArn = "arn:aws:iam::516651178292:role/inventory-admin-testing"

var integrationHosts = []string{
	"host1",
	"host2",
}

var integrationSecrets = []string{
	"secret1",
	"secret2",
}

var integrationServices = []string{
	"service1",
	"service2",
}

type clientIntegrationTest struct {
	client             *Client
	db                 dynamodbiface.DynamoDBAPI
	heartbeatCallerArn string
	adminCreds         *credentials.Credentials
	testPrefix         string
	cb                 IterationHandler
	calls              []*ConsumedSecret
}

func newClientIntegrationTest(t *testing.T) (cit *clientIntegrationTest) {
	cfg := &Config{
		TableName: integrationTestTableName,
	}

	awsConfig := &aws.Config{
		Region: aws.String(integrationTestRegion),
	}

	stsclient := sts.New(session.New(awsConfig))
	arp := &stscreds.AssumeRoleProvider{
		Duration:     900 * time.Second,
		ExpiryWindow: 10 * time.Second,
		RoleARN:      integrationStormwatchRoleArn,
		Client:       stsclient,
	}
	creds := credentials.NewCredentials(arp)
	awsConfig.WithCredentials(creds)

	sess := session.New(awsConfig)
	adminCreds := adminCreds(t)
	adminSession := adminSession(t, adminCreds)

	stsclient = sts.New(adminSession)
	identity, err := stsclient.GetCallerIdentity(&sts.GetCallerIdentityInput{})
	if err != nil {
		t.Fatal(err)
	}

	cit = &clientIntegrationTest{
		client:             New(cfg, sess),
		db:                 dynamodb.New(adminSession),
		heartbeatCallerArn: aws.StringValue(identity.Arn),
		adminCreds:         adminCreds,
		testPrefix:         uuid.NewV4().String(),
	}

	cit.cb = func(cs *ConsumedSecret) (err error) {
		cit.calls = append(cit.calls, cs)
		return
	}

	return
}

func adminCreds(t *testing.T) (creds *credentials.Credentials) {
	stsclient := sts.New(session.New(&aws.Config{
		Region: aws.String(integrationTestRegion),
	}))
	arp := &stscreds.AssumeRoleProvider{
		Duration:     900 * time.Second,
		ExpiryWindow: 10 * time.Second,
		RoleARN:      integrationAdminRoleArn,
		Client:       stsclient,
	}
	creds = credentials.NewCredentials(arp)
	return
}

func adminSession(t *testing.T, creds *credentials.Credentials) (sess *session.Session) {
	sess = session.New(&aws.Config{
		Region:      aws.String(integrationTestRegion),
		Credentials: creds,
	})
	return
}

func (cit *clientIntegrationTest) resetCallback() {
	cit.calls = []*ConsumedSecret{}

	cit.cb = func(cs *ConsumedSecret) (err error) {
		cit.calls = append(cit.calls, cs)
		return
	}
}

func (cit *clientIntegrationTest) setup(t *testing.T) {
	cit.putHeartbeatCombos(t)
}

func (cit *clientIntegrationTest) teardown(t *testing.T) {
	cit.deleteHeartbeats(t)
}

func (cit *clientIntegrationTest) putHeartbeatCombos(t *testing.T) {
	cfg := &heartbeat.Config{
		URL:    "https://o1ll5pv4hh.execute-api.us-west-2.amazonaws.com/testing",
		Region: integrationTestRegion,
	}

	for _, host := range integrationHosts {
		for _, secret := range integrationSecrets {
			for _, service := range integrationServices {
				cfg.Service = cit.testPrefix + service
				cfg.Host = cit.testPrefix + host

				hb := heartbeat.New(cit.adminCreds, cfg, nil)
				hb.UpdateHeartbeat(&heartbeat.Secret{
					Name:      cit.testPrefix + secret,
					UpdatedAt: 1948,
				})

				hb.SendHeartbeat()
			}
		}
	}
}

func (cit *clientIntegrationTest) deleteHeartbeat(t *testing.T, host, secret, service string) {
	compositeKey := cit.getCompositeKey(t, host, secret, service)

	_, err := cit.db.DeleteItem(&dynamodb.DeleteItemInput{
		TableName: aws.String(integrationTestTableName),
		Key: map[string]*dynamodb.AttributeValue{
			"composite_key": {S: aws.String(compositeKey)},
		},
	})
	assert.NoError(t, err)
}

func (cit *clientIntegrationTest) deleteHeartbeats(t *testing.T) {
	for _, host := range integrationHosts {
		for _, secret := range integrationSecrets {
			for _, service := range integrationServices {
				cit.deleteHeartbeat(t, cit.testPrefix+host, cit.testPrefix+secret, cit.testPrefix+service)
			}
		}
	}
}

func (cit *clientIntegrationTest) getCompositeKey(t *testing.T, host, secret, service string) (key string) {
	key = cit.heartbeatCallerArn
	key += ":" + host
	key += ":" + service
	key += ":" + secret
	return
}

func TestClientIntegration(t *testing.T) {
	cit := newClientIntegrationTest(t)
	cit.setup(t)
	defer cit.teardown(t)

	t.Run("query for existing host", func(t *testing.T) {
		defer cit.resetCallback()
		testHost := cit.testPrefix + "host1"

		assert := assert.New(t)

		err := cit.client.IterateByHost(testHost, cit.cb)
		assert.NoError(err)

		assert.Len(cit.calls, 4)
		for _, consumer := range cit.calls {
			assert.Equal(testHost, consumer.Host)
			assert.NotEmpty(consumer.HeartbeatReceived)
		}
	})

	t.Run("query for existing secret", func(t *testing.T) {
		defer cit.resetCallback()
		secretName := cit.testPrefix + "secret1"

		assert := assert.New(t)

		err := cit.client.IterateBySecret(secretName, cit.cb)
		assert.NoError(err)

		assert.Len(cit.calls, 4)
		for _, consumer := range cit.calls {
			assert.Equal(secretName, consumer.Secret)
			assert.NotEmpty(consumer.HeartbeatReceived)
		}
	})

	t.Run("query for existing service", func(t *testing.T) {
		defer cit.resetCallback()
		serviceName := cit.testPrefix + "service1"

		assert := assert.New(t)

		err := cit.client.IterateByService(serviceName, cit.cb)
		assert.NoError(err)

		assert.Len(cit.calls, 4)
		for _, consumer := range cit.calls {
			assert.Equal(serviceName, consumer.Service)
			assert.NotEmpty(consumer.HeartbeatReceived)
		}
	})

	t.Run("query for non existing host", func(t *testing.T) {
		defer cit.resetCallback()
		assert := assert.New(t)
		err := cit.client.IterateByHost("muh-entropy", cit.cb)
		assert.Empty(cit.calls)
		assert.NoError(err)
	})

	t.Run("query for non existing secret", func(t *testing.T) {
		defer cit.resetCallback()
		assert := assert.New(t)
		err := cit.client.IterateBySecret("derp", cit.cb)
		assert.Empty(cit.calls)
		assert.NoError(err)
	})

	t.Run("query for non existing service", func(t *testing.T) {
		defer cit.resetCallback()
		assert := assert.New(t)
		err := cit.client.IterateByService("derp", cit.cb)
		assert.Empty(cit.calls)
		assert.NoError(err)
	})
}
