package consumedsecrets

import (
	"testing"

	"code.justin.tv/systems/sandstorm/inventory/heartbeat"

	"github.com/stretchr/testify/assert"
)

type withOrderedDedupeTest struct {
	calls []*ConsumedSecret
	cb    IterationHandler
}

func newWithOrderedDedupeTest() (wodt *withOrderedDedupeTest) {
	wodt = &withOrderedDedupeTest{
		calls: []*ConsumedSecret{},
	}
	wodt.cb = WithOrderedDedupe(func(cs *ConsumedSecret) (err error) {
		wodt.calls = append(wodt.calls, cs)
		return
	})
	return
}

func TestWithOrderedDedupe(t *testing.T) {
	t.Run("callback called on new heartbeat", func(t *testing.T) {
		assert := assert.New(t)
		wodt := newWithOrderedDedupeTest()

		err := wodt.cb(&ConsumedSecret{
			DynamoHeartbeat: heartbeat.DynamoHeartbeat{
				Secret:    "mySecretName",
				UpdatedAt: 1,
				Service:   "myService",
				Host:      "myHost",
			},
		})

		expected := []*ConsumedSecret{
			&ConsumedSecret{
				DynamoHeartbeat: heartbeat.DynamoHeartbeat{
					Secret:    "mySecretName",
					UpdatedAt: 1,
					Service:   "myService",
					Host:      "myHost",
				},
			},
		}

		assert.NoError(err)
		assert.Equal(expected, wodt.calls)
	})

	t.Run("callback called on newer heartbeat", func(t *testing.T) {
		assert := assert.New(t)
		wodt := newWithOrderedDedupeTest()

		err := wodt.cb(&ConsumedSecret{
			DynamoHeartbeat: heartbeat.DynamoHeartbeat{
				Secret:    "mySecretName",
				UpdatedAt: 1,
				Host:      "myHost",
				Service:   "myService",
			},
		})

		assert.NoError(err)

		err = wodt.cb(&ConsumedSecret{
			DynamoHeartbeat: heartbeat.DynamoHeartbeat{
				Secret:    "mySecretName",
				UpdatedAt: 2,
				Host:      "myHost",
				Service:   "myService",
			},
		})

		expected := []*ConsumedSecret{
			&ConsumedSecret{
				DynamoHeartbeat: heartbeat.DynamoHeartbeat{
					Secret:    "mySecretName",
					UpdatedAt: 1,
					Host:      "myHost",
					Service:   "myService",
				},
			},
			&ConsumedSecret{
				DynamoHeartbeat: heartbeat.DynamoHeartbeat{
					Secret:    "mySecretName",
					UpdatedAt: 2,
					Host:      "myHost",
					Service:   "myService",
				},
			},
		}

		assert.NoError(err)
		assert.Equal(expected, wodt.calls)
	})

	t.Run("callback not called on older heartbeat", func(t *testing.T) {
		assert := assert.New(t)
		wodt := newWithOrderedDedupeTest()

		err := wodt.cb(&ConsumedSecret{
			DynamoHeartbeat: heartbeat.DynamoHeartbeat{
				Secret:    "mySecretName",
				UpdatedAt: 2,
				Service:   "myService",
				Host:      "myHost",
			},
		})

		assert.NoError(err)

		err = wodt.cb(&ConsumedSecret{
			DynamoHeartbeat: heartbeat.DynamoHeartbeat{
				Secret:    "mySecretName",
				UpdatedAt: 1,
				Service:   "myService",
				Host:      "myHost",
			},
		})

		expected := []*ConsumedSecret{
			&ConsumedSecret{
				DynamoHeartbeat: heartbeat.DynamoHeartbeat{
					Secret:    "mySecretName",
					UpdatedAt: 2,
					Service:   "myService",
					Host:      "myHost",
				},
			},
		}

		assert.NoError(err)
		assert.Equal(expected, wodt.calls)
	})
}
