package heartbeat

import (
	"bytes"
	"context"
	"encoding/json"
	"io/ioutil"
	"net/http"
	"net/url"
	"testing"
	"time"

	"code.justin.tv/systems/sandstorm/mocks"

	"github.com/aws/aws-sdk-go/aws/credentials"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/mock"
)

type clientTest struct {
	client         *Client
	mockHTTPClient *mocks.HTTPClient
}

func newClientTest() (ct *clientTest) {
	config := &Config{
		Service: "TestService",
		Host:    "TestHost",
	}

	client := New(credentials.NewEnvCredentials(), config, nil)

	mockHTTPClient := new(mocks.HTTPClient)
	client.httpClient = mockHTTPClient

	ct = &clientTest{
		client:         client,
		mockHTTPClient: mockHTTPClient,
	}
	return
}

func (ct *clientTest) teardown(t *testing.T) {
	ct.mockHTTPClient.AssertExpectations(t)
}

func TestInterface(t *testing.T) {
	t.Run("Client should implement interface", func(t *testing.T) {
		assert := assert.New(t)
		var api API
		api = &Client{}
		assert.NotNil(api)
	})
}

func TestNewSetsDefaultsConfig(t *testing.T) {
	t.Run("test default configs", func(t *testing.T) {
		assert := assert.New(t)

		creds := credentials.NewEnvCredentials()

		client := New(creds, &Config{}, nil)

		assert.Equal(defaultURL, client.Config.URL, "unexpected default url set")
		assert.Equal(defaultInterval, client.Config.Interval, "unexpected default interval set")
		assert.NotNil(client.httpClient, "no http client set")
		assert.NotNil(client.heartbeatState, "heartbeat state not initialized")
	})
}

func TestSendHeartbeat(t *testing.T) {
	t.Run("put should not be called if no secrets", func(t *testing.T) {
		ct := newClientTest()
		defer ct.teardown(t)

		ct.client.SendHeartbeat()
	})

	t.Run("put should called if there are secrets", func(t *testing.T) {
		ct := newClientTest()
		defer ct.teardown(t)

		ct.client.UpdateHeartbeat(&Secret{
			Name:      "mySecretName",
			UpdatedAt: 123456,
		})

		res := &http.Response{
			StatusCode: http.StatusOK,
			Body:       ioutil.NopCloser(bytes.NewReader(nil)),
		}

		ct.mockHTTPClient.On("Do", mock.Anything).Return(res, nil).Once()

		ct.client.SendHeartbeat()
	})
}

func TestUpdateHeartbeat(t *testing.T) {
	t.Run("update heartbeat updates the current heartbeat state", func(t *testing.T) {
		assert := assert.New(t)
		invClient := New(credentials.NewEnvCredentials(), &Config{}, nil)

		secretName := "test"
		assert.Equal(0, len(invClient.heartbeatState.secretsToReport), "invalid number of secrets to report")

		secret := &Secret{
			Name:      secretName,
			UpdatedAt: 123456,
		}
		invClient.UpdateHeartbeat(secret)
		assert.Equal(1, len(invClient.heartbeatState.secretsToReport), "invalid number of secrets to report")

		//Check FetchedAt timestamp
		ts := time.Now().Unix()
		assert.True(invClient.heartbeatState.secretsToReport[secretName].FetchedAt <= ts, "invalid retrieved at timestamp set")
		assert.True(invClient.heartbeatState.secretsToReport[secretName].FetchedAt > 0, "retrieved at timestamp was not set")
	})
}

func TestBuildPutHeartbeatRequest(t *testing.T) {
	t.Run("test buildPutHeartbeatRequest", func(t *testing.T) {
		config := &Config{
			Service: "TestService",
			Host:    "TestHost",
			FQDN:    "TestFQDN",
		}

		t.Run("Ensure request is has no secret.", func(t *testing.T) {
			assert := assert.New(t)
			invClient := New(credentials.NewEnvCredentials(), config, nil)

			req, err := invClient.buildPutHeartbeatRequest()
			if assert.NoError(err) {
				expectedURL, err := url.Parse(defaultURL + "/heartbeat")
				assert.NoError(err)

				assert.Equal("PUT", req.Method, "Invalid request method")
				assert.Equal(expectedURL, req.URL)
				assert.Equal(headerContentTypeJSON, req.Header.Get(httpHeaderContentType), "invalid content type set in req")

				actualHeartbeat := &heartbeat{}
				err = json.NewDecoder(req.Body).Decode(actualHeartbeat)
				if err != nil {
					t.Fatal(err)
				}
				expectedHeartbeat := &heartbeat{
					Host:    config.Host,
					FQDN:    config.FQDN,
					Service: config.Service,
				}
				assert.Equal(expectedHeartbeat, actualHeartbeat)
			}

		})
		t.Run("Request body contains 1 secret", func(t *testing.T) {
			assert := assert.New(t)
			invClient := New(credentials.NewEnvCredentials(), config, nil)

			secretName := "secretName"
			secret := &Secret{
				Name:      secretName,
				UpdatedAt: 123456,
			}
			invClient.UpdateHeartbeat(secret)
			req, err := invClient.buildPutHeartbeatRequest()
			assert.NoError(err)

			actualHeartbeat := &heartbeat{}
			err = json.NewDecoder(req.Body).Decode(actualHeartbeat)
			if err != nil {
				t.Fatal(err)
			}
			expectedHeartbeat := &heartbeat{
				Host:    config.Host,
				FQDN:    config.FQDN,
				Service: config.Service,
				Secrets: []*Secret{secret},
			}
			assert.Equal(expectedHeartbeat, actualHeartbeat)

		})
	})
}

func TestPutHeartbeat(t *testing.T) {
	config := &Config{
		Service: "TestService",
		Host:    "TestHost",
	}
	t.Run("Returns success when making a valid request.", func(t *testing.T) {
		assert := assert.New(t)
		invClient := New(credentials.NewEnvCredentials(), config, nil)
		mockedHTTPClient := new(mocks.HTTPClient)
		invClient.httpClient = mockedHTTPClient
		httpResponse := &http.Response{StatusCode: http.StatusOK, Body: ioutil.NopCloser(bytes.NewBuffer([]byte{}))}
		mockedHTTPClient.On("Do", mock.Anything).Return(httpResponse, nil).Once()

		err := invClient.putHeartbeat()
		assert.NoError(err)
		mockedHTTPClient.AssertExpectations(t)
	})
}

func TestFlushHeartbeat(t *testing.T) {
	config := &Config{
		Service: "TestService",
		Host:    "TestHost",
	}
	invClient := New(credentials.NewEnvCredentials(), config, nil)

	ctx, cancel := context.WithTimeout(context.Background(), time.Second)
	defer cancel()

	invClient.FlushHeartbeat(ctx)

	received := false
	select {
	case <-invClient.flushChan:
		received = true
	case <-ctx.Done():
	}
	assert.True(t, received)
}
