from datetime import datetime
import os

from botocore.errorfactory import ClientError
import boto3

ENV_TABLE_NAME_KEY = 'INVENTORY_TABLE_NAME'
ENV_TTL_KEY = 'INVENTORY_TTL'
DEFAULT_TTL = 24 * 3600


def put_heartbeat(event, context):
    """
    event.body_json should have the following format:

        {
            "host": "",
            "fqdn": "",
            "service": "",
            "instance_id": "",
            "manager_version": "",
            "secrets": [
                {
                    "name": "test",
                    "updated_at":12345566,
                    ..........
                }
            ]
        }

    """
    db = boto3.client('dynamodb')

    try:
        requests = _parse_requests(event, context)
    except KeyError as e:
        return bad_request(e)

    for kws in requests:
        try:
            _update_heartbeat(db=db, **kws)
        except ClientError as e:
            errcode = e.response['Error']['Code']
            if errcode != 'ConditionalCheckFailedException':
                raise e
            _put_heartbeat(db=db, **kws)


def bad_request(e):
    return {
        'errorMessage': '[BadRequest] {}'.format(e),
    }


def _parse_requests(event, context):
    host = event['body_json']['host']
    fqdn = event['body_json'].get('fqdn', host)
    service = event['body_json']['service']
    instance_id = event['body_json'].get('instance_id'),
    manager_version = event['body_json'].get('manager_version'),

    return [{
        'host': host,
        'fqdn': fqdn,
        'service': service,
        'secret': secret,
        'instance_id': instance_id,
        'manager_version': manager_version,
    } for secret in event['body_json']['secrets']]

def _update_heartbeat(db, host, fqdn, service, secret, instance_id, manager_version):
    update_keys = (
        'secret',
        'host',
        'fqdn',
        'service',
        'fetched_at',
        'updated_at',
        # sent by us
        'expires_at',
        'heartbeat_received',
    )

    now = int(datetime.utcnow().timestamp())

    update_expression = list(map(lambda k: '{k} = :{k}'.format(k=k), update_keys))
    expression_attribute_values = {
        ':secret': {'S': secret['name']},
        ':host': {'S': host},
        ':fqdn': {'S': fqdn},
        ':service': {'S': service},
        ':fetched_at': {'N': str(secret['fetched_at'])},
        ':updated_at': {'N': str(secret['updated_at'])},
        ':expires_at': {'N': str(now + _get_ttl())},
        ':heartbeat_received': {'N': str(now)},
    }

    if instance_id:
        update_expression.append('instance_id = :instance_id')
        expression_attribute_values['instance_id'] = {'S': instance_id}

    if manager_version:
        update_expression.append('manager_version = :manager_version')
        expression_attribute_values['manager_version'] = {'S': manager_version}

    update_expression = 'SET ' + ','.join(update_expression)
    db.update_item(
        TableName=_get_table_name(),
        Key={
            'composite_key': {'S': _heartbeat_key(
                host=host,
                service=service,
                secret=secret,
            )},
        },
        UpdateExpression=update_expression,
        ExpressionAttributeValues=expression_attribute_values,
        ConditionExpression='attribute_exists(composite_key)',
    )


def _put_heartbeat(db, host, fqdn,service, secret, instance_id, manager_version):
    now = int(datetime.utcnow().timestamp())
    item = {
        'composite_key': {'S': _heartbeat_key(
            host=host,
            service=service,
            secret=secret,
        )},
        'secret': {'S': secret['name']},
        'host': {'S': host},
        'fqdn': {'S': fqdn},
        'service': {'S': service},
        'fetched_at': {'N': str(secret['fetched_at'])},
        'updated_at': {'N': str(secret['updated_at'])},
        'first_retrieved_at': {'N': str(secret['fetched_at'])},
        'expires_at': {'N': str(now + _get_ttl())},
        'heartbeat_received': {'N': str(now)},
    }
    if instance_id:
        item['instance_id'] = {'S': instance_id}
    if manager_version:
        item['manager_version'] = {'S': manager_version}

    db.put_item(
        TableName=_get_table_name(),
        Item=item,
        ConditionExpression='attribute_not_exists(composite_key)',
    )


def _heartbeat_key(host, service, secret):
    return ':'.join((host, service, secret['name']))


def _get_table_name():
    return os.environ[ENV_TABLE_NAME_KEY]


def _get_ttl():
    """ Return a TTL as a integer from the environment
    """
    return int(os.environ.get(ENV_TTL_KEY, DEFAULT_TTL))
