from datetime import datetime
import mock
import os
import unittest

from botocore.errorfactory import ClientError
import boto3
import pytest

import main

TEST_HOST = 'myTestHost'
TEST_FQDN = 'myTestFQDN'
TEST_SERVICE = 'myTestService'
TEST_SECRET_NAME = 'mySecretName'
TEST_UPDATED_AT = '483'
TEST_INSTANCE_ID = "someRandomeUUID"
TEST_MANAGER_VERSION = "1.0"

TEST_ADMIN_ROLE = 'arn:aws:iam::516651178292:role/inventory-admin-testing'  # NOQA
TEST_LAMBDA_ROLE = 'arn:aws:iam::516651178292:role/inventory-lambda-service-testing'  # NOQA


class TestHeartbeatIntegration(unittest.TestCase):
    table_name = 'heartbeats-testing'

    @classmethod
    def setUpClass(cls):
        os.environ[main.ENV_TABLE_NAME_KEY] = cls.table_name

        # first, we want to assume the admin role and also generate credentials
        # for the lambda user
        sts = boto3.client('sts')
        admin_creds = sts.assume_role(
            RoleArn=TEST_ADMIN_ROLE,
            RoleSessionName='TestAdminRole',
        )['Credentials']

        sts = boto3.client(
            'sts',
            region_name='us-west-2',
            aws_access_key_id=admin_creds['AccessKeyId'],
            aws_secret_access_key=admin_creds['SecretAccessKey'],
            aws_session_token=admin_creds['SessionToken']
        )
        lambda_creds = sts.assume_role(
            RoleArn=TEST_LAMBDA_ROLE,
            RoleSessionName='TestLambdaRole',
        )['Credentials']

        cls.db = boto3.client(
            'dynamodb',
            region_name='us-west-2',
            aws_access_key_id=lambda_creds['AccessKeyId'],
            aws_secret_access_key=lambda_creds['SecretAccessKey'],
            aws_session_token=lambda_creds['SessionToken']
        )

        cls.admin_db = boto3.client(
            'dynamodb',
            region_name='us-west-2',
            aws_access_key_id=admin_creds['AccessKeyId'],
            aws_secret_access_key=admin_creds['SecretAccessKey'],
            aws_session_token=admin_creds['SessionToken']
        )

    def tearDown(self):
        self.delete_heartbeat()

    def get_key(self):
        key = main._heartbeat_key(
            host=TEST_HOST,
            service=TEST_SERVICE,
            secret={'name': TEST_SECRET_NAME},
        )
        return {'composite_key': {'S': key}}

    def get_heartbeat(self):
        res = self.admin_db.get_item(
            ConsistentRead=True,
            TableName=self.table_name,
            Key=self.get_key(),
        )
        return res['Item']

    def delete_heartbeat(self):
        return self.admin_db.delete_item(
            TableName=self.table_name,
            Key=self.get_key(),
        )

    def update_heartbeat(self, fetched_at):
        main._update_heartbeat(
            self.db,
            host=TEST_HOST,
            fqdn=TEST_FQDN,
            service=TEST_SERVICE,
            manager_version = TEST_MANAGER_VERSION,
            instance_id = TEST_INSTANCE_ID,
            secret={
                'name': TEST_SECRET_NAME,
                'updated_at': TEST_UPDATED_AT,
                'fetched_at': fetched_at,
            },
        )

    def get_expected(self, fetched_at, expiration, first_retrieved_at,
                     heartbeat_received):
        return dict({
            'secret': {'S': TEST_SECRET_NAME},
            'host': {'S': TEST_HOST},
            'fqdn': {'S': TEST_FQDN},
            'service': {'S': TEST_SERVICE},
            'updated_at': {'N': TEST_UPDATED_AT},
            'fetched_at': {'N': str(fetched_at)},
            'first_retrieved_at': {'N': str(first_retrieved_at)},
            'expires_at': {'N': str(expiration)},
            'heartbeat_received': {'N': str(heartbeat_received)},
        }, **self.get_key())

    def test_put_then_update_should_succeed(self):
        fetched_at = 487

        kws = {
            'host': TEST_HOST,
            'fqdn': TEST_FQDN,
            'service': TEST_SERVICE,
            'manager_version': None,
            'instance_id': None,
            'secret': {
                'name': TEST_SECRET_NAME,
                'updated_at': 483,
                'fetched_at': fetched_at,
            },
        }

        now = datetime.utcnow()
        with mock.patch('main.datetime') as _datetime:
            _datetime.utcnow.return_value = now
            main._put_heartbeat(self.db, **kws)

        assert self.get_expected(
            fetched_at=fetched_at,
            heartbeat_received=int(now.timestamp()),
            expiration=main.DEFAULT_TTL + int(now.timestamp()),
            first_retrieved_at=fetched_at,
        ) == self.get_heartbeat()

        td = 50
        now = datetime.utcnow()
        with mock.patch('main.datetime') as _datetime:
            _datetime.utcnow.return_value = now
            main._update_heartbeat(
                self.db,
                **dict(kws, secret={
                    'name': TEST_SECRET_NAME,
                    'updated_at': TEST_UPDATED_AT,
                    'fetched_at': fetched_at + td,
                }))

        assert self.get_expected(
            fetched_at=fetched_at + td,
            heartbeat_received=int(now.timestamp()),
            expiration=main.DEFAULT_TTL + int(now.timestamp()),
            first_retrieved_at=fetched_at,
        ) == self.get_heartbeat()

    def test_double_put_failure(self):
        kws = {
            'host': TEST_HOST,
            'fqdn': TEST_FQDN,
            'service': TEST_SERVICE,
            'instance_id': TEST_INSTANCE_ID,
            'manager_version': None,
            'secret': {
                'name': TEST_SECRET_NAME,
                'updated_at': 485,
                'fetched_at': 487,
            },
        }

        main._put_heartbeat(self.db, **kws)
        with pytest.raises(ClientError):
            main._put_heartbeat(self.db, **kws)

    def test_update_without_put_failure(self):
        with pytest.raises(ClientError):
            main._update_heartbeat(
                self.db,
                host=TEST_HOST,
                fqdn=TEST_FQDN,
                instance_id=TEST_INSTANCE_ID,
                manager_version=TEST_MANAGER_VERSION,
                service=TEST_SERVICE,
                secret={
                    'name': TEST_SECRET_NAME,
                    'updated_at': 483,
                    'fetched_at': 487,
                })


class TestPutHeartbeat(unittest.TestCase):
    def setUp(self):
        self.mockedBoto3 = mock.patch('main.boto3').start()

    def tearDown(self):
        mock.patch.stopall()

    def test_initial_put_ok(self):
        secret = {
            'name': 'mySecretName',
            'fetched_at': 184,
            'updated_at': 188,
        }

        put_item = self.mockedBoto3.client.return_value.put_item
        update_item = self.mockedBoto3.client.return_value.update_item

        update_item.return_value = None
        main.put_heartbeat({
            'body_json': {
                'host': TEST_HOST,
                'fqdn': TEST_FQDN,
                'service': TEST_SERVICE,
                'secrets': [secret],
            },
        }, {})

        assert update_item.call_count == 1
        assert put_item.call_count == 0

    def test_initial_put_ok_no_fqdn(self):
        secret = {
            'name': 'mySecretName',
            'fetched_at': 184,
            'updated_at': 188,
        }

        put_item = self.mockedBoto3.client.return_value.put_item
        update_item = self.mockedBoto3.client.return_value.update_item

        update_item.return_value = None
        main.put_heartbeat({
            'body_json': {
                'host': TEST_HOST,
                'service': TEST_SERVICE,
                'instance_id': TEST_INSTANCE_ID,
                'manager_version': TEST_MANAGER_VERSION,
                'secrets': [secret],
            },
        }, {})

        assert update_item.call_count == 1
        assert put_item.call_count == 0

    def test_initial_put_with_conditional_check_failure(self):
        secret = {
            'name': 'mySecretName',
            'fetched_at': 184,
            'updated_at': 188,
        }

        put_item = self.mockedBoto3.client.return_value.put_item
        update_item = self.mockedBoto3.client.return_value.update_item

        e = ClientError(
            error_response={'Error': {
                'Code': 'ConditionalCheckFailedException',
            }},
            operation_name='',
        )
        update_item.side_effect = e

        update_item.return_value = None
        main.put_heartbeat({
            'body_json': {
                'host': TEST_HOST,
                'fqdn': TEST_FQDN,
                'service': TEST_SERVICE,
                'secrets': [secret],
                'instance_id': TEST_INSTANCE_ID,
                'manager_version': TEST_MANAGER_VERSION,
            },
        }, {})

        assert update_item.call_count == 1
        assert put_item.call_count == 1

    def test_initial_put_with_other_failure(self):
        secret = {
            'name': 'mySecretName',
            'fetched_at': 184,
            'updated_at': 188,
        }

        put_item = self.mockedBoto3.client.return_value.put_item
        update_item = self.mockedBoto3.client.return_value.update_item

        e = ClientError(
            error_response={'Error': {
                'Code': 'Something Else',
            }},
            operation_name='',
        )
        update_item.side_effect = e

        update_item.return_value = None

        with pytest.raises(ClientError):
            main.put_heartbeat({
                'body_json': {
                    'host': TEST_HOST,
                    'fqdn': TEST_FQDN,
                    'service': TEST_SERVICE,
                    'secrets': [secret],
                    'instance_id': TEST_INSTANCE_ID,
                    'manager_version': TEST_MANAGER_VERSION,
                },
            }, {})

        assert update_item.call_count == 1
        assert put_item.call_count == 0

    def test_empty_body_json_failure(self):
        response = main.put_heartbeat({
            'body_json': {},
        }, {})

        assert response == {
            'errorMessage': '[BadRequest] \'host\'',
        }
