package fanout

import (
	"time"

	"code.justin.tv/feeds/clients/feeddataflow"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/graphdb/proto/graphdb"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"golang.org/x/net/context"
)

type FollowsDiscoveryConfig struct {
	Disabled *distconf.Bool
}

func (f *FollowsDiscoveryConfig) Load(d *distconf.Distconf) error {
	f.Disabled = d.Bool("candidatediscovery.follows.disabled", false)
	return nil
}

// FollowsDiscovery is a ConsumerCandidateDiscovery whose candidates are a user's follows
type FollowsDiscovery struct {
	GraphDB graphdb.GraphDB
	Stats   *service_common.StatSender
	Log     *log.ElevatedLog
	Config  *FollowsDiscoveryConfig
}

var _ ConsumerCandidateDiscovery = &FollowsDiscovery{}

// Candidates returns a user's follows
func (d *FollowsDiscovery) Candidates(ctx context.Context, in *Activity) ([]FeedWithMetadata, error) {
	if in.Entity.Namespace() == entity.NamespaceFollow {
		// As an efficiency thing, we don't broadcast follows to people that follow you
		return nil, nil
	}
	return d.userFollows(ctx, in.Actor.ID())
}

func (d *FollowsDiscovery) userFollows(ctx context.Context, userID string) ([]FeedWithMetadata, error) {
	d.Log.DebugCtx(ctx, "actor", userID, "Detecting follows")
	if d.Config.Disabled.Get() {
		d.Log.Debug("Friendship discovery disabled")
		return nil, nil
	}
	cands := make([]FeedWithMetadata, 0, 1024)
	cursor := ""
	for {
		startTime := time.Now()
		res, err := d.GraphDB.EdgeList(
			ctx,
			&graphdb.EdgeListRequest{
				From: &graphdb.Node{
					Id:   userID,
					Type: "user",
				},
				EdgeType: "followed_by",
				Page: &graphdb.PagedRequest{
					Cursor: cursor,
					Limit:  2000,
				},
			},
		)
		if err != nil {
			d.Stats.IncC("Cohesion.ListAssoc.err", 1, 1.0)
			return nil, errors.Wrap(err, "follows list failed")
		}
		d.Stats.TimingDurationC("Cohesion.ListAssoc", time.Since(startTime), 1.0)

		now := time.Now()
		for _, cell := range res.Edges {
			M := feeddataflow.Metadata{}
			M.SetFollow(cell.Edge.Edge.To.Id, userID, true, now)
			cands = append(cands, FeedWithMetadata{
				Feed:     "n:" + cell.Edge.Edge.To.Id,
				Metadata: &M,
			})
			if len(cands)%100000 == 0 {
				d.Log.DebugCtx(ctx, "len", len(cands), "user_id", userID, "debugging memory leak")
			}
		}
		d.Log.DebugCtx(ctx, "items", len(res.Edges), "total_so_far", len(cands), "fetch finished")
		cursor = res.Cursor
		if cursor == "" {
			break
		}
	}
	return cands, nil
}
