package fanout

import (
	"time"

	"code.justin.tv/chat/friendship/app/api/responses"
	friendship "code.justin.tv/chat/friendship/client"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/foundation/twitchclient"
	"golang.org/x/net/context"
)

// FriendshipClient asserts what functions are needed from friendship.Client to allow mocking
type FriendshipClient interface {
	GetFriends(ctx context.Context, userID string, params friendship.GetFriendsParams, reqOpts *twitchclient.ReqOpts) (*responses.FriendListResponse, error)
}

type FriendsDiscoveryConfig struct {
	Disabled *distconf.Bool
}

func (f *FriendsDiscoveryConfig) Load(d *distconf.Distconf) error {
	f.Disabled = d.Bool("candidatediscovery.friends.disabled", false)
	return nil
}

var _ FriendshipClient = friendship.Client(nil)

// FriendsDiscovery is a ConsumerCandidateDiscovery whose candidates are a user's friends
type FriendsDiscovery struct {
	Friendship FriendshipClient
	Stats      *service_common.StatSender
	Log        *log.ElevatedLog
	Config     *FriendsDiscoveryConfig
}

var _ ConsumerCandidateDiscovery = &FriendsDiscovery{}

// Candidates returns a user's friends
func (d *FriendsDiscovery) Candidates(ctx context.Context, in *Activity) ([]FeedWithMetadata, error) {
	return d.userFriends(ctx, in.Actor.ID())
}

func (d *FriendsDiscovery) userFriends(ctx context.Context, userID string) ([]FeedWithMetadata, error) {
	d.Log.Debug("actor", userID, "Detecting friends")
	if d.Config.Disabled.Get() {
		d.Log.Debug("Friendship discovery disabled")
		return nil, nil
	}

	startTime := time.Now()
	res, err := d.Friendship.GetFriends(ctx, userID, friendship.GetFriendsParams{}, nil)
	if err != nil {
		d.Stats.IncC("FriendshipClient.GetFriends.err", 1, 1.0)
		return nil, errors.Wrapf(err, "friendship call failed for %s", userID)
	}
	d.Stats.TimingDurationC("FriendshipClient.GetFriends", time.Since(startTime), 1.0)

	cands := make([]FeedWithMetadata, len(res.Friends))
	for i, friend := range res.Friends {
		cands[i] = FeedWithMetadata{
			Feed: "n:" + friend.Entity.ID,
		}
	}

	return cands, nil
}
