package fanout

import (
	"expvar"
	"fmt"
	"io"

	"code.justin.tv/feeds/clients/feeddataflow"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-common/verb"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	"github.com/aws/aws-sdk-go/aws/awsutil"
	"golang.org/x/net/context"
)

// FeedDemultiplexer accepts feed stories, determines who to send them to, and groups them and sends them to
// a source.
type FeedDemultiplexer struct {
	ActivityProcessor   ActivityProcessor
	GroupedFeedReceiver FeedStoriesReceiver
}

// FeedWithMetadata combines a feed ID with metadata that ranked items for that feed may need.
// It is combined with the final result is sent to masonry.
type FeedWithMetadata struct {
	Feed     string
	Metadata *feeddataflow.Metadata
}

// ConsumerCandidateDiscovery is an object that can find users that are interested in a feed story
type ConsumerCandidateDiscovery interface {
	Candidates(ctx context.Context, in *Activity) ([]FeedWithMetadata, error)
}

// ActivityBatchDiscovery can create ad hoc batches of stories for an activity
type ActivityBatchDiscovery interface {
	ActivityBatches(ctx context.Context, in *Activity) (*ActivityBatch, error)
}

// FeedStoriesReceiver can receive a batch of feed stories
type FeedStoriesReceiver interface {
	QueueLowPriorityFeedStories(ctx context.Context, stories *ActivityBatch) error
	QueueMidPriorityFeedStories(ctx context.Context, stories *ActivityBatch) error
	QueueHighPriorityFeedStories(ctx context.Context, stories *ActivityBatch) error
}

// ActivityDestination can receive feed stories
type ActivityDestination interface {
	AddActivity(ctx context.Context, i *Activity) error
}

// Activity is the feed object that comes into fanout service
type Activity struct {
	Entity   entity.Entity          `json:"entity"`
	Verb     verb.Verb              `json:"verb"`
	Actor    entity.Entity          `json:"actor"`
	Metadata *feeddataflow.Metadata `json:"metadata,omitempty"`
}

func (i *Activity) String() string {
	return fmt.Sprintf("Actor:[%s] Verb:[%s] Entity:[%s]", i.Actor, i.Verb, i.Entity)
}

// ActivityBatch groups a number of activities with feed ids that should process those activities
type ActivityBatch struct {
	Activities []*Activity
	FeedIDs    []FeedWithMetadata
}

func (f *ActivityBatch) processSize() int64 {
	return int64(len(f.Activities) * len(f.FeedIDs))
}

// Format prints activityBatch to satisfy fmt.Formatter
func (f *ActivityBatch) Format(s fmt.State, c rune) {
	if c == 'v' {
		fmt.Fprint(s, awsutil.Prettify(f.FeedIDs)+"|"+awsutil.Prettify(f.Activities))
		return
	}
	fmt.Fprint(s, f.String())
}

func (f *ActivityBatch) String() string {
	return fmt.Sprintf("FeedStory:[%d] len(Consumers):[%d]", len(f.Activities), len(f.FeedIDs))
}

func (f *ActivityBatch) split(maxSize int64) (*ActivityBatch, *ActivityBatch) {
	if len(f.Activities) == 0 || len(f.FeedIDs) == 0 {
		return nil, nil
	}
	if f.processSize() <= maxSize {
		return f, nil
	}
	batchActivities := len(f.Activities) < len(f.FeedIDs)
	if batchActivities {
		feedIdsToKeep := maxSize / int64(len(f.Activities))
		if feedIdsToKeep == 0 {
			feedIdsToKeep = 1
		}
		newBatch := ActivityBatch{
			Activities: f.Activities,
			FeedIDs:    f.FeedIDs[0:feedIdsToKeep],
		}
		leftover := ActivityBatch{
			Activities: f.Activities,
			FeedIDs:    f.FeedIDs[feedIdsToKeep:],
		}
		return &newBatch, &leftover
	}

	activitiesToKeep := maxSize / int64(len(f.FeedIDs))
	if activitiesToKeep == 0 {
		activitiesToKeep = 1
	}
	newBatch := ActivityBatch{
		FeedIDs:    f.FeedIDs,
		Activities: f.Activities[0:activitiesToKeep],
	}
	leftover := ActivityBatch{
		FeedIDs:    f.FeedIDs,
		Activities: f.Activities[activitiesToKeep:],
	}
	return &newBatch, &leftover
}

var _ ActivityDestination = &FeedDemultiplexer{}

// ActivitySource is an object that gives fanout feed stories
type ActivitySource interface {
	io.Closer
	Start() error
}

// Var is a expvar that is the service itself in JSON form
func (d *FeedDemultiplexer) Var() expvar.Var {
	return expvar.Func(func() interface{} {
		return d
	})
}

// AddActivity will detect edges of a message, group them, and send them out in groups
func (d *FeedDemultiplexer) AddActivity(ctx context.Context, i *Activity) error {
	return d.ActivityProcessor.AddActivity(ctx, i, d.GroupedFeedReceiver)
}

// filterActivites is a temporary need: If we ever want story cards about "X followed Y", then we'll need to
// remove this
func filterActivites(activities []*Activity) []*Activity {
	unfilteredActivites := make([]*Activity, 0, len(activities))
	for _, activity := range activities {
		if activity.Entity.Namespace() == entity.NamespaceFollow || activity.Entity.Namespace() == entity.NamespaceFriend {
			continue
		}
		unfilteredActivites = append(unfilteredActivites, activity)
	}
	return unfilteredActivites
}

// ActivityProcessor processes activity but does not have the FeedStoriesReceiver that receives the events
type ActivityProcessor struct {
	Log            *log.ElevatedLog
	EdgeDetectors  []ConsumerCandidateDiscovery
	BatchDiscovery []ActivityBatchDiscovery
	MaxBatchSize   *distconf.Int
	Stats          *service_common.StatSender
}

func (d *ActivityProcessor) sendToCorrectQueue(ctx context.Context, toSend *ActivityBatch, groupedMsg *ActivityBatch, recv FeedStoriesReceiver) error {
	if toSend.processSize() == 1 {
		if err := recv.QueueHighPriorityFeedStories(ctx, toSend); err != nil {
			return err
		}
		d.Stats.IncC("hipri", 1, 1.0)
	} else if groupedMsg == nil {
		if err := recv.QueueMidPriorityFeedStories(ctx, toSend); err != nil {
			return err
		}
		d.Stats.IncC("medpri", 1, 1.0)
	} else {
		if err := recv.QueueLowPriorityFeedStories(ctx, toSend); err != nil {
			return err
		}
		d.Stats.IncC("lowpri", 1, 1.0)
	}
	return nil
}

func (d *ActivityProcessor) sendBatch(ctx context.Context, recv FeedStoriesReceiver, groupedMsg *ActivityBatch) error {
	if groupedMsg != nil {
		groupedMsg.Activities = filterActivites(groupedMsg.Activities)
	}
	for groupedMsg != nil {
		var toSend *ActivityBatch
		toSend, groupedMsg = groupedMsg.split(d.MaxBatchSize.Get())
		if groupedMsg == nil && toSend == nil {
			return nil
		}
		if toSend == nil {
			panic("Logic error.  toSend should never end up nil")
		}
		d.Log.DebugCtx(ctx, "num_feed_ids", len(toSend.FeedIDs), "activities", len(toSend.Activities), "Sending some feed stories")
		if err := d.sendToCorrectQueue(ctx, toSend, groupedMsg, recv); err != nil {
			return err
		}
	}
	return nil
}

func (d *ActivityProcessor) processBatchDiscovery(ctx context.Context, recv FeedStoriesReceiver, i *Activity) error {
	for _, detector := range d.BatchDiscovery {
		batches, err := detector.ActivityBatches(ctx, i)
		if err != nil {
			return errors.Wrap(err, "Cannot find batches for story")
		}
		if err := d.sendBatch(ctx, recv, batches); err != nil {
			return err
		}
	}
	return nil
}

func (d *ActivityProcessor) processEdgeDetectors(ctx context.Context, recv FeedStoriesReceiver, i *Activity) error {
	for _, detector := range d.EdgeDetectors {
		edges, err := detector.Candidates(ctx, i)
		if err != nil {
			return errors.Wrap(err, "cannot find candidates")
		}
		d.Log.DebugCtx(ctx, "num_edges", len(edges), "Found some edges")
		if len(edges) == 0 {
			continue
		}
		batch := ActivityBatch{
			Activities: []*Activity{i},
			FeedIDs:    edges,
		}
		if err := d.sendBatch(ctx, recv, &batch); err != nil {
			return err
		}
	}
	return nil
}

// AddActivity will process activity into stories and send them to recv
func (d *ActivityProcessor) AddActivity(ctx context.Context, i *Activity, recv FeedStoriesReceiver) error {
	ctx = d.Log.NormalLog.Dims.Append(ctx, "a.actor", i.Actor.Encode(), "a.ent", i.Entity.Encode(), "a.verb", i.Verb)
	d.Log.DebugCtx(ctx, "ActivityProcessor detected a new message")
	if err := d.processEdgeDetectors(ctx, recv, i); err != nil {
		return err
	}
	return d.processBatchDiscovery(ctx, recv, i)
}
