package fanout

import (
	"code.justin.tv/feeds/clients/duplo"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-common/verb"
	"code.justin.tv/feeds/service-common"
	"golang.org/x/net/context"
)

// FollowerStoryBatches replays follow and unfollow actions, as well as friend actions
type FollowerStoryBatches struct {
	DuploClient *duplo.Client
	Stats       *service_common.StatSender
}

var _ ActivityBatchDiscovery = &FollowerStoryBatches{}

// ActivityBatches will replay duplo posts by user ID for follow/unfollow actions
func (f *FollowerStoryBatches) ActivityBatches(ctx context.Context, in *Activity) (*ActivityBatch, error) {
	for _, prefix := range []string{entity.NamespaceFollow, entity.NamespaceFriend} {
		ab, err := f.activityBatchesForPrefix(ctx, in, prefix)
		if err != nil {
			return nil, err
		}
		if ab != nil {
			return ab, nil
		}
	}
	return nil, nil
}

func (f *FollowerStoryBatches) activityBatchesForPrefix(ctx context.Context, in *Activity, prefix string) (*ActivityBatch, error) {
	if in.Entity.Namespace() != prefix {
		return nil, nil
	}
	f.Stats.IncC("fbatch."+in.Entity.Namespace(), 1, 1.0)

	// Actor just followed/friended target.  We put target's posts into actor's news feed.
	ret := &ActivityBatch{
		FeedIDs:    []FeedWithMetadata{{Feed: "n:" + in.Actor.ID()}},
		Activities: make([]*Activity, 0),
	}
	userID := in.Entity.ID()

	// See CF-870.  Replaying every post was putting very high load on duplo, so we're limiting it to 10 posts
	getIDsResp, err := f.DuploClient.GetPostIDsByUser(ctx, userID, &duplo.GetPostIDsByUserOptions{
		Limit: 10,
	})
	if err != nil {
		return nil, errors.Wrap(err, "unable to find stories")
	}

	for _, postID := range getIDsResp.PostIDs {
		ret.Activities = append(ret.Activities, &Activity{
			Actor:  entity.New(entity.NamespaceUser, userID),
			Verb:   verb.Create,
			Entity: entity.New(entity.NamespacePost, postID),
		})
	}

	if len(ret.Activities) == 0 {
		return nil, nil
	}

	return ret, nil
}
