package fanout

import (
	"encoding/json"
	"io"
	"strings"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-common/verb"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/service-common/feedsqs"
	"github.com/aws/aws-sdk-go/service/sqs"
	"golang.org/x/net/context"
)

/**
{
  "Type" : "Notification",
  "MessageId" : "2bcb59d5-f77b-52bc-838a-092536ca290a",
  "TopicArn" : "arn:aws:sns:us-west-2:603200399373:friendship_production_events",
  "Message" : "{\"requester_id\":\"111754530\",\"target_id\":\"141859672\"}",
  "Timestamp" : "2017-01-11T23:27:45.821Z",
  "SignatureVersion" : "1",
  "Signature" : "Cx8oKIcq310wkBeqWd4LKOwUWM3zX9fGgvM+aBcXPHe201htAWqKOXDF5lZijKCQeyxWogH1PWCIJ7PiVrnHz3s9A2we2N9TgtyXGY50bu40pm553iIEqoa778YDsEkjANzyHvHTv85nngTZIkncqhQNMzIOez4qxjz+n8YielQLnbvXo/dcVHyi4Z5jgt0ILdQ8JfyNAoX9PaCGIbImJBM/R50MyrBAIeJ70G3D57HjqiQW/qFKBEahjXMitDFhIYGUUZcaBJ9O2wlotXLdyGiTTQDCAVlT8EmSPiRiM2cgpCQm0yX10QRTPMd/1PqntPSvG0Xei3B/eMHxAu9XPg==",
  "SigningCertURL" : "https://sns.us-west-2.amazonaws.com/SimpleNotificationService-b95095beb82e8f6a046b3aafc7f4149a.pem",
  "UnsubscribeURL" : "https://sns.us-west-2.amazonaws.com/?Action=Unsubscribe&SubscriptionArn=arn:aws:sns:us-west-2:603200399373:friendship_production_events:8a117309-ee9a-4c37-bbb1-5008f3e52169",
  "MessageAttributes" : {
    "event" : {"Type":"String","Value":"friend_request_accepted"}
  }
}
*/

// Should match friendship.AcceptFriendRequestEvent and friendship.RemoveFriendEvent at
// https://git-aws.internal.justin.tv/chat/friendship/blob/master/client/sns_payloads.go
const friendshipAcceptedMessageType = "accept_friend_request"
const friendshipRemovedMessageType = "remove_friend"

type snsInputMessage struct {
	Type              string
	MessageID         string `json:"MessageId"`
	TopicArn          string
	Message           string
	Timestamp         string
	SignatureVersion  string
	Signature         string
	Subject           string
	SigningCertURL    string
	UnsubscribeURL    string
	MessageAttributes map[string]messageAttribute
}

type messageAttribute struct {
	Type  string
	Value string
}

// Should match friendship.AcceptFriendRequestPayload and friendship.RemoveFriendPayload at
// https://git-aws.internal.justin.tv/chat/friendship/blob/master/client/sns_payloads.go
type friendshipMessage struct {
	UserID       string `json:"user_id"`
	TargetUserID string `json:"target_user_id"`
}

// FriendshipSqsSourceConfig configures FriendshipSqsSource
type FriendshipSqsSourceConfig struct {
	feedsqs.SQSQueueProcessorConfig
	enableFriendshipMessages *distconf.Bool
}

func friendshipEventFromAttributes(attr map[string]messageAttribute) (string, error) {
	eventKey, exists := attr["event"]
	if !exists {
		return "", errors.New("could not find event key in message")
	}
	if eventKey.Type != "String" {
		return "", errors.New("invalid type for event key: " + eventKey.Type)
	}
	return eventKey.Value, nil
}

// Load configuration information
func (c *FriendshipSqsSourceConfig) Load(dconf *distconf.Distconf) error {
	c.enableFriendshipMessages = dconf.Bool("fanout.enable-friendship-msgs", true)
	return c.SQSQueueProcessorConfig.Verify(dconf, "friendship-activity", time.Second*3, 8, time.Minute*15)
}

// FriendshipSqsSource can receive messages from a SQS queue about friendship events, from the friendship service
type FriendshipSqsSource struct {
	feedsqs.SQSQueueProcessor
	Destination      ActivityDestination
	FriendshipConfig *FriendshipSqsSourceConfig
}

// Setup sets ProcessMessage and calls sub Setup()
func (s *FriendshipSqsSource) Setup() error {
	s.SQSQueueProcessor.ProcessMessage = s.processMessage
	return s.SQSQueueProcessor.Setup()
}

func parseFriendshipMessage(in io.Reader) (friendshipMessage, string, error) {
	msg := &snsInputMessage{}
	if err := json.NewDecoder(in).Decode(msg); err != nil {
		return friendshipMessage{}, "", err
	}
	var fm friendshipMessage
	if err := json.NewDecoder(strings.NewReader(msg.Message)).Decode(&fm); err != nil {
		return friendshipMessage{}, "", err
	}
	friendshipEvent, err := friendshipEventFromAttributes(msg.MessageAttributes)
	if err != nil {
		return friendshipMessage{}, "", err
	}
	return fm, friendshipEvent, nil
}

func (s *FriendshipSqsSource) processFriendshipMessage(ctx context.Context, msgBody io.Reader) error {
	fm, event, err := parseFriendshipMessage(msgBody)
	if err != nil {
		return err
	}
	var activityVerb verb.Verb
	if event == friendshipAcceptedMessageType {
		s.Stats.IncC("friendship.accepted", 1, 1.0)
		activityVerb = verb.Create
	} else if event == friendshipRemovedMessageType {
		s.Stats.IncC("friendship.removed", 1, 1.0)
		activityVerb = verb.Delete
	} else {
		s.Stats.IncC("friendship.unknown", 1, 1.0)
		// Unknown type
		return nil
	}
	if !s.FriendshipConfig.enableFriendshipMessages.Get() {
		return nil
	}

	// Two friend activities to cross populate posts
	activity1 := &Activity{
		Entity: entity.New(entity.NamespaceFriend, fm.UserID),
		Verb:   activityVerb,
		Actor:  entity.New(entity.NamespaceUser, fm.TargetUserID),
	}
	activity2 := &Activity{
		Entity: entity.New(entity.NamespaceFriend, fm.TargetUserID),
		Verb:   activityVerb,
		Actor:  entity.New(entity.NamespaceUser, fm.UserID),
	}
	return service_common.ConsolidateErrors([]error{
		s.Destination.AddActivity(ctx, activity1),
		s.Destination.AddActivity(ctx, activity2),
	})
}

func (s *FriendshipSqsSource) processMessage(ctx context.Context, m *sqs.Message) error {
	if m.Body != nil {
		return s.processFriendshipMessage(ctx, strings.NewReader(*m.Body))
	}
	return nil
}
