package fanout

import (
	"strings"
	"testing"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-common/verb"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/service-common/feedsqs"
	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
)

const exampleMessage = `{
  "Type" : "Notification",
  "MessageId" : "2bcb59d5-f77b-52bc-838a-092536ca290a",
  "TopicArn" : "arn:aws:sns:us-west-2:603200399373:friendship_production_events",
  "Message" : "{\"user_id\":\"111754530\",\"target_user_id\":\"141859672\"}",
  "Timestamp" : "2017-01-11T23:27:45.821Z",
  "SignatureVersion" : "1",
  "Signature" : "Cx8oKIcq310wkBeqWd4LKOwUWM3zX9fGgvM+aBcXPHe201htAWqKOXDF5lZijKCQeyxWogH1PWCIJ7PiVrnHz3s9A2we2N9TgtyXGY50bu40pm553iIEqoa778YDsEkjANzyHvHTv85nngTZIkncqhQNMzIOez4qxjz+n8YielQLnbvXo/dcVHyi4Z5jgt0ILdQ8JfyNAoX9PaCGIbImJBM/R50MyrBAIeJ70G3D57HjqiQW/qFKBEahjXMitDFhIYGUUZcaBJ9O2wlotXLdyGiTTQDCAVlT8EmSPiRiM2cgpCQm0yX10QRTPMd/1PqntPSvG0Xei3B/eMHxAu9XPg==",
  "SigningCertURL" : "https://sns.us-west-2.amazonaws.com/SimpleNotificationService-b95095beb82e8f6a046b3aafc7f4149a.pem",
  "UnsubscribeURL" : "https://sns.us-west-2.amazonaws.com/?Action=Unsubscribe&SubscriptionArn=arn:aws:sns:us-west-2:603200399373:friendship_production_events:8a117309-ee9a-4c37-bbb1-5008f3e52169",
  "MessageAttributes" : {
    "event" : {"Type":"String","Value":"accept_friend_request"}
  }
}`

func TestParseFriendshipMessage(t *testing.T) {
	Convey("Messages should parse", t, func() {
		msg, eventType, err := parseFriendshipMessage(strings.NewReader(exampleMessage))
		So(err, ShouldBeNil)
		So(eventType, ShouldEqual, friendshipAcceptedMessageType)
		So(msg.UserID, ShouldEqual, "111754530")
		So(msg.TargetUserID, ShouldEqual, "141859672")
	})
}

type memDst struct {
	buf []*Activity
}

func (m *memDst) AddActivity(ctx context.Context, i *Activity) error {
	m.buf = append(m.buf, i)
	return nil
}

func TestFriendshipSqsSource(t *testing.T) {
	Convey("Messages should forward", t, func() {
		memoryDst := &memDst{}
		ctx := context.Background()
		conf := FriendshipSqsSourceConfig{}
		memDistconf := distconf.InMemory{}
		So(memDistconf.Write("friendship-activity.sqssource.queue_url", []byte("_ignore_")), ShouldBeNil)
		So(conf.Load(&distconf.Distconf{
			Readers: []distconf.Reader{
				&memDistconf,
			},
		}), ShouldBeNil)
		x := FriendshipSqsSource{
			Destination:      memoryDst,
			FriendshipConfig: &conf,
			SQSQueueProcessor: feedsqs.SQSQueueProcessor{
				Stats: service_common.NopStatSender(),
			},
		}
		So(x.processFriendshipMessage(ctx, strings.NewReader(exampleMessage)), ShouldBeNil)
		a := memoryDst.buf[0]
		So(a.Actor, ShouldResemble, entity.New(entity.NamespaceUser, "141859672"))
		So(a.Verb, ShouldEqual, verb.Create)
		So(a.Entity, ShouldResemble, entity.New("friend", "111754530"))
	})
}
