package fanout

import (
	"net/http"

	"encoding/json"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/service-common"
	"goji.io"
	"goji.io/pat"
	"golang.org/x/net/context"
)

// HTTPConfig configures HTTPServer
type HTTPConfig struct {
	service_common.BaseHTTPServerConfig
}

// Load load config values from distconf
func (c *HTTPConfig) Load(dconf *distconf.Distconf) error {
	return c.Verify(dconf, "fanout")
}

// ProcessorActivityDestination can receive feed stories and where to send them
type ProcessorActivityDestination interface {
	AddActivity(ctx context.Context, i *Activity, recv FeedStoriesReceiver) error
}

// MemoryFeedStoriesReceiver stories activities in memory to return them as JSON
type MemoryFeedStoriesReceiver struct {
	Low  []*ActivityBatch
	Mid  []*ActivityBatch
	High []*ActivityBatch
}

var _ FeedStoriesReceiver = &MemoryFeedStoriesReceiver{}

// QueueLowPriorityFeedStories saves the story to the Low array
func (m *MemoryFeedStoriesReceiver) QueueLowPriorityFeedStories(ctx context.Context, stories *ActivityBatch) error {
	m.Low = append(m.Low, stories)
	return nil
}

// QueueMidPriorityFeedStories saves the story to the Mid array
func (m *MemoryFeedStoriesReceiver) QueueMidPriorityFeedStories(ctx context.Context, stories *ActivityBatch) error {
	m.Mid = append(m.Mid, stories)
	return nil
}

// QueueHighPriorityFeedStories saves the story to the High array
func (m *MemoryFeedStoriesReceiver) QueueHighPriorityFeedStories(ctx context.Context, stories *ActivityBatch) error {
	m.High = append(m.High, stories)
	return nil
}

// HTTPServer allows debugging and sending manual fanout objects
type HTTPServer struct {
	service_common.BaseHTTPServer
	Destination       ActivityDestination
	CustomDestination ProcessorActivityDestination
}

// SetupRoutes configures the http server
func (h *HTTPServer) SetupRoutes(mux *goji.Mux) {
	mux.Handle(pat.Post("/private/activity"), h.createHandler("private_activity", h.privatePostActivity))
	mux.Handle(pat.Post("/private/simulate_activity"), h.createHandler("private_activity", h.privateSimulateActivity))
	mux.Handle(pat.Get("/debug/health"), h.createHandler("health_check", h.healthCheck))
}

func (h *HTTPServer) createHandler(name string, callback func(req *http.Request) (interface{}, error)) http.Handler {
	return h.XRay.Handler(&service_common.JSONHandler{
		Log:          h.Log,
		Stats:        h.Stats.NewSubStatSender(name),
		ItemCallback: callback,
	})
}

func (h *HTTPServer) healthCheck(r *http.Request) (interface{}, error) {
	return "OK", nil
}

func (h *HTTPServer) privatePostActivity(req *http.Request) (interface{}, error) {
	msg := &Activity{}
	err := json.NewDecoder(req.Body).Decode(msg)
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "cannot parse request body as JSON"),
		}
	}
	if msg.Actor.ID() == "" || msg.Entity.ID() == "" || msg.Verb == "" {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("expect non empty actor, entity, and verb"),
		}
	}
	return "OK", h.Destination.AddActivity(req.Context(), msg)
}

func (h *HTTPServer) privateSimulateActivity(req *http.Request) (interface{}, error) {
	msg := &Activity{}
	err := json.NewDecoder(req.Body).Decode(msg)
	if err != nil {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.Wrap(err, "cannot parse request body as JSON"),
		}
	}
	if msg.Actor.ID() == "" || msg.Entity.ID() == "" || msg.Verb == "" {
		return nil, &service_common.CodedError{
			Code: http.StatusBadRequest,
			Err:  errors.New("expect non empty actor, entity, and verb"),
		}
	}
	m := MemoryFeedStoriesReceiver{}
	if err := h.CustomDestination.AddActivity(req.Context(), msg, &m); err != nil {
		return "", err
	}
	return m, nil
}
