package fanout

import (
	"time"

	"fmt"

	"code.justin.tv/feeds/clients/feeddataflow"
	"code.justin.tv/feeds/clients/masonry"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/service-common"
	"golang.org/x/net/context"
)

// SqsFeedReceiver can send story batches to an SQS queue
type SqsFeedReceiver struct {
	HighPriQueueClient *masonry.QueueClient
	MidPriQueueClient  *masonry.QueueClient
	LowPriQueueClient  *masonry.QueueClient
	Stats              *service_common.StatSender
}

// QueueLowPriorityFeedStories sends a story to the low pri queue
func (s *SqsFeedReceiver) QueueLowPriorityFeedStories(ctx context.Context, stories *ActivityBatch) error {
	return s.queueFeedStories(ctx, stories, s.LowPriQueueClient, "low")
}

// QueueMidPriorityFeedStories sends the story to mid pri SQS
func (s *SqsFeedReceiver) QueueMidPriorityFeedStories(ctx context.Context, stories *ActivityBatch) error {
	return s.queueFeedStories(ctx, stories, s.MidPriQueueClient, "mid")
}

// QueueHighPriorityFeedStories sends the story to high pri SQS
func (s *SqsFeedReceiver) QueueHighPriorityFeedStories(ctx context.Context, stories *ActivityBatch) error {
	return s.queueFeedStories(ctx, stories, s.HighPriQueueClient, "high")
}

func (s *SqsFeedReceiver) queueFeedStories(ctx context.Context, stories *ActivityBatch, client *masonry.QueueClient, level string) error {
	startTime := time.Now()
	aggregateMetadata := feeddataflow.Metadata{}
	for _, act := range stories.Activities {
		aggregateMetadata.MergeFrom(act.Metadata)
	}
	feedIDs := make([]string, 0, len(stories.FeedIDs))
	for _, feed := range stories.FeedIDs {
		aggregateMetadata.MergeFrom(feed.Metadata)
		feedIDs = append(feedIDs, feed.Feed)
	}
	masonryStoryBatch := masonry.ActivityBatch{
		Metadata:   &aggregateMetadata,
		FeedIDs:    feedIDs,
		Activities: make([]*masonry.Activity, 0, len(stories.Activities)),
	}
	for _, activity := range stories.Activities {
		masonryStoryBatch.Activities = append(masonryStoryBatch.Activities, &masonry.Activity{
			Entity: activity.Entity,
			Verb:   activity.Verb,
			Actor:  entity.New(entity.NamespaceUser, activity.Actor.ID()),
			// Don't put metadata here.  Leave it at the aggregate level
		})
	}
	if err := client.QueueActivityBatch(ctx, &masonryStoryBatch); err != nil {
		s.Stats.IncC(fmt.Sprintf("%s.AddFeedStories_error", level), stories.processSize(), 1)
		return errors.Wrap(err, "cannot send feed stories")
	}
	s.Stats.TimingDurationC(fmt.Sprintf("%s.AddFeedStories", level), time.Since(startTime), 1)
	s.Stats.IncC(fmt.Sprintf("%s.msg_output", level), stories.processSize(), 1)
	return nil
}
