package fanout

import (
	"encoding/json"
	"io"
	"strings"
	"time"

	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-common/verb"
	"code.justin.tv/feeds/service-common/feedsqs"
	"github.com/aws/aws-sdk-go/service/sqs"
	"golang.org/x/net/context"
)

/**
{
  "Type" : "Notification",
  "MessageId" : "73f33b0d-4b01-5480-9596-6a3d536280f8",
  "TopicArn" : "arn:aws:sns:us-west-2:914569885343:following-service_production_updatefollow_notifications",
  "Message" : "{\"from_id\":\"116256076\",\"target_id\":\"133038144\",\"event\":\"unfollow\",\"updated_at\":\"2017-03-01T00:30:27.927773484Z\"}",
  "Timestamp" : "2017-03-01T00:30:27.930Z",
  "SignatureVersion" : "1",
  "Signature" : "cv1Se9VVd4Wui+TsVn54gkRIQrlPenG+GuIiCX+VzXPNidgctfxYoPSQ2oeevOYExwmFgRKwkMS5uCtmAMbybBwOaiN5ojTZIVyaRMN6GQ2alxfLn27msFi8arYcpo+FCOU9/DQrEhFIp+u/2Wi0aHpwnEaOLiE6GJeG0V2qd5YPT2H+tEw8k8ErwDeFrsrWGOgmM48Vz05uj/CGpoI4rkKdnL7wjp1DD7oJBZ2ZKMCRvTxJv9ttL2eaDl+Cwsjsy+iFXQcmIXbe3QTSIMuWri5i+XrnY8fQUvAb0zXBZ9NdN+aSxb4OGOQPNiccBHhmt9KAI14ANL0hT0FNc/t3Hw==",
  "SigningCertURL" : "https://sns.us-west-2.amazonaws.com/SimpleNotificationService-b95095beb82e8f6a046b3aafc7f4149a.pem",
  "UnsubscribeURL" : "https://sns.us-west-2.amazonaws.com/?Action=Unsubscribe&SubscriptionArn=arn:aws:sns:us-west-2:914569885343:following-service_production_updatefollow_notifications:00c04c1f-86a0-4d0d-9897-1a855bce429f",
  "MessageAttributes" : {
    "event" : {"Type":"String","Value":"updatefollow"}
  }
}
*/

type updateFollowMessage struct {
	FromID   string `json:"from_id"`
	TargetID string `json:"target_id"`
	Event    string `json:"event"`
}

// UpdateFollowSqsSourceConfig configures UpdateFollowSqsSource
type UpdateFollowSqsSourceConfig struct {
	feedsqs.SQSQueueProcessorConfig
	enableUpdateFollowMessages *distconf.Bool
}

// Load configuration information
func (c *UpdateFollowSqsSourceConfig) Load(dconf *distconf.Distconf) error {
	c.enableUpdateFollowMessages = dconf.Bool("fanout.enable-updatefollow-msgs", true)
	return c.SQSQueueProcessorConfig.Verify(dconf, "update-follow", time.Second*3, 8, time.Minute*15)
}

// UpdateFollowSqsSource can receive messages from a SQS queue about follow events, from the following-service
type UpdateFollowSqsSource struct {
	feedsqs.SQSQueueProcessor
	Destination        ActivityDestination
	UpdateFollowConfig *UpdateFollowSqsSourceConfig
}

// Setup sets ProcessMessage and calls sub Setup()
func (s *UpdateFollowSqsSource) Setup() error {
	s.SQSQueueProcessor.ProcessMessage = s.processMessage
	return s.SQSQueueProcessor.Setup()
}

func parseUpdateFollowMessage(in io.Reader) (updateFollowMessage, error) {
	msg := &snsInputMessage{}
	if err := json.NewDecoder(in).Decode(msg); err != nil {
		return updateFollowMessage{}, err
	}
	var updateMsg updateFollowMessage
	if err := json.NewDecoder(strings.NewReader(msg.Message)).Decode(&updateMsg); err != nil {
		return updateFollowMessage{}, err
	}
	return updateMsg, nil
}

func (s *UpdateFollowSqsSource) processUpdateFollowMessage(ctx context.Context, msgBody io.Reader) error {
	fm, err := parseUpdateFollowMessage(msgBody)
	if err != nil {
		return err
	}
	if !s.UpdateFollowConfig.enableUpdateFollowMessages.Get() {
		return nil
	}

	var v verb.Verb

	if fm.Event == "unfollow" {
		v = verb.Delete
	} else {
		v = verb.Create
	}

	activity := &Activity{
		Entity: entity.New(entity.NamespaceFollow, fm.TargetID),
		Verb:   v,
		Actor:  entity.New(entity.NamespaceUser, fm.FromID),
	}

	return s.Destination.AddActivity(ctx, activity)
}

func (s *UpdateFollowSqsSource) processMessage(ctx context.Context, m *sqs.Message) error {
	if m.Body != nil {
		return s.processUpdateFollowMessage(ctx, strings.NewReader(*m.Body))
	}
	return nil
}
