package main

import (
	"expvar"
	"fmt"
	"net"
	"net/http"
	"os"
	"os/signal"

	"code.justin.tv/chat/friendship/client"
	"code.justin.tv/feeds/clients/duplo"
	"code.justin.tv/feeds/clients/masonry"
	"code.justin.tv/feeds/ctxlog"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/expvar2"
	"code.justin.tv/feeds/fanout/cmd/fanout/internal/fanout"
	"code.justin.tv/feeds/graphdb/proto/graphdb"
	"code.justin.tv/feeds/metrics/sfx/sfxstatsd"
	"code.justin.tv/feeds/metrics/statsdim"
	"code.justin.tv/feeds/service-common"
	"code.justin.tv/feeds/service-common/feedsqs"
	"code.justin.tv/foundation/twitchclient"
	"code.justin.tv/hygienic/httpheaders"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sqs"
)

const (
	serviceName = "fanout"
	teamName    = "feeds"
)

func sfxStastdConfig() sfxstatsd.SetupConfig {
	return sfxstatsd.SetupConfig{
		CounterMetricRules: []*statsdim.ConfigurableDelimiterMetricRule{
			{
				MetricPath:    "sqs_source.*.msg_input",
				DimensionsMap: "%.queue_name.%",
			},
			{
				MetricPath:    "sqs_source.*.msg_input",
				DimensionsMap: "%.queue_name.%",
			},
			{
				MetricPath:    "feed_demultiplexer.*",
				DimensionsMap: "%.queue_name",
				MetricName:    "sent_batches",
			},
			{
				MetricPath:    "sqs_receiver.*.msg_output",
				DimensionsMap: "%.queue_priority.%",
				MetricName:    "count",
			},
			{
				MetricPath:    "follow_batch.fbatch.follow",
				DimensionsMap: "%.%.entity_namespace",
				MetricName:    "count",
			},
			{
				MetricPath:    "sqs_source.*.friendship.*",
				DimensionsMap: "%.queue_name.%.action_type",
				MetricName:    "friendship_queue_counts",
			},
			{
				MetricPath:    "http.*.code.*",
				DimensionsMap: "-.handler.-.http_code",
				MetricName:    "http.hits",
			},
			{
				MetricPath:    "follows_discovery.Cohesion.*",
				DimensionsMap: "%.%.method",
				MetricName:    "method_count",
			},
		},
		TimingMetricRules: []*statsdim.ConfigurableDelimiterMetricRule{
			{
				MetricPath:    "sqs_source.*.sqs.*",
				DimensionsMap: "%.queue_name.%.method",
				MetricName:    "method_timing",
			},
			{
				MetricPath:    "sqs_source.*.*.time",
				DimensionsMap: "%.queue_name..%",
				MetricName:    "method_timing",
			},
			{
				MetricPath:    "sqs_receiver.*.*",
				DimensionsMap: "%.queue_priority.method",
				MetricName:    "method_times",
			},
			{
				MetricPath:    "http.*.time",
				DimensionsMap: "%.handler.%",
			},
			{
				MetricPath:    "follows_discovery.Cohesion.*",
				DimensionsMap: "%.%.method",
				MetricName:    "method_time",
			},
		},
	}
}

var instance = service{
	osExit: os.Exit,
	serviceCommon: service_common.ServiceCommon{
		ConfigCommon: service_common.ConfigCommon{
			Team:       teamName,
			Service:    serviceName,
			OsGetenv:   os.Getenv,
			OsHostname: os.Hostname,
		},
		CodeVersion:    CodeVersion,
		SfxSetupConfig: sfxStastdConfig(),
	},
}

// CodeVersion is set by go build.  Should be the SHA1 of the code when it was built.
var CodeVersion string

type service struct {
	osExit func(code int)

	destination fanout.FeedStoriesReceiver
	feedMux     fanout.FeedDemultiplexer

	runner              service_common.ServiceRunner
	onListen            func(net.Addr)
	server              fanout.HTTPServer
	serviceCommon       service_common.ServiceCommon
	sqsSource           fanout.SqsSource
	followsDiscovery    fanout.FollowsDiscovery
	friendshipDiscovery fanout.FriendsDiscovery
	sigChan             chan os.Signal
	awsSetup            func(*session.Session, []*aws.Config) error

	graphDBClient    graphdb.GraphDB
	friendshipClient friendship.Client

	configs struct {
		config                        fanoutConfig
		HTTPConfig                    fanout.HTTPConfig
		SqsSourceConfig               fanout.SqsSourceConfig
		SqsReceiverHighPriorityConfig masonry.HighPriorityQueueConfig
		SqsReceiverMidPriorityConfig  masonry.MidPriorityQueueConfig
		SqsReceiverLowPriorityConfig  masonry.LowPriorityQueueConfig
		duploConfig                   duplo.Config
		friendshipSqsSourceConfig     fanout.FriendshipSqsSourceConfig
		updateFollowSqsSourceConfig   fanout.UpdateFollowSqsSourceConfig
		friendsDiscoveryConfig        fanout.FriendsDiscoveryConfig
		followsDiscoveryConfig        fanout.FollowsDiscoveryConfig
	}
}

type fanoutConfig struct {
	graphdbURL    *distconf.Str
	friendshipURL *distconf.Str
}

func (c *fanoutConfig) Load(dconf *distconf.Distconf) error {
	c.graphdbURL = dconf.Str("graphdb.url", "")
	if c.graphdbURL.Get() == "" {
		return errors.New("graphdb.url")
	}
	c.friendshipURL = dconf.Str("candidatediscovery.friends.friendship_url", "")
	if c.friendshipURL.Get() == "" {
		return errors.New("unable to find a valid candidatediscovery.friends.friendship_url")
	}
	return nil
}

func (f *service) setupGraphDBClient() error {
	if f.graphDBClient != nil {
		return nil
	}
	addr := f.configs.config.graphdbURL.Get()
	client := httpheaders.WithHeader(http.DefaultClient, "X-Caller-Service", "fanout")
	f.graphDBClient = graphdb.NewGraphDBProtobufClient(addr, client)
	return nil
}

func (f *service) setup() error {
	f.serviceCommon.ExpvarHandler.Exported = make(map[string]expvar.Var)

	expvar2.AddStrings(f.serviceCommon.ExpvarHandler.Exported, map[string]string{
		"rollbar_page":  fmt.Sprintf("https://rollbar.com/Twitch/%s/", serviceName),
		"github_link":   fmt.Sprintf("https://git-aws.internal.justin.tv/%s/%s/tree/%s", teamName, serviceName, CodeVersion),
		"graphana_page": fmt.Sprintf("https://grafana.internal.justin.tv/dashboard/db/%s", serviceName),
		"build_page":    fmt.Sprintf("https://jenkins.internal.justin.tv/job/%s-%s/", teamName, serviceName),
		"deploy_page":   fmt.Sprintf("https://clean-deploy.internal.justin.tv/#/%s/%s", teamName, serviceName),
	})

	if err := f.serviceCommon.Setup(); err != nil {
		return err
	}
	if f.awsSetup != nil {
		if err := f.awsSetup(service_common.CreateAWSSession(f.serviceCommon.Config)); err != nil {
			return err
		}
	}

	if err := service_common.LoadConfigs(
		f.serviceCommon.Config, &f.configs.config, &f.configs.HTTPConfig, &f.configs.SqsSourceConfig, &f.configs.SqsReceiverHighPriorityConfig,
		&f.configs.SqsReceiverMidPriorityConfig, &f.configs.SqsReceiverLowPriorityConfig, &f.configs.duploConfig, &f.configs.friendshipSqsSourceConfig,
		&f.configs.updateFollowSqsSourceConfig, &f.configs.friendsDiscoveryConfig, &f.configs.followsDiscoveryConfig); err != nil {
		return err
	}

	if err := f.setupGraphDBClient(); err != nil {
		return err
	}

	var err error
	f.friendshipClient, err = friendship.NewClient(twitchclient.ClientConf{Host: f.configs.config.friendshipURL.Get()})
	return err
}

func (f *service) inject() {
	duploClient := duplo.Client{
		Config: &f.configs.duploConfig,
		RequestDoer: &ctxlog.LoggedDoer{
			Logger: f.serviceCommon.Log,
			C:      &f.serviceCommon.Ctxlog,
			Client: &http.Client{},
		},
		NewHTTPRequest: ctxlog.WrapHTTPRequestWithCtxlog(&f.serviceCommon.Ctxlog, http.NewRequest),
	}
	session, awsConf := service_common.CreateAWSSession(f.serviceCommon.Config)
	f.destination = &fanout.SqsFeedReceiver{
		HighPriQueueClient: &masonry.QueueClient{
			Sqs:    sqs.New(session, awsConf...),
			Config: &f.configs.SqsReceiverHighPriorityConfig.QueueConfig,
			Ch:     &f.serviceCommon.Ctxlog,
			Log:    f.serviceCommon.Log,
		},
		MidPriQueueClient: &masonry.QueueClient{
			Sqs:    sqs.New(session, awsConf...),
			Config: &f.configs.SqsReceiverMidPriorityConfig.QueueConfig,
			Ch:     &f.serviceCommon.Ctxlog,
			Log:    f.serviceCommon.Log,
		},
		LowPriQueueClient: &masonry.QueueClient{
			Sqs:    sqs.New(session, awsConf...),
			Config: &f.configs.SqsReceiverLowPriorityConfig.QueueConfig,
			Ch:     &f.serviceCommon.Ctxlog,
			Log:    f.serviceCommon.Log,
		},
		Stats: &service_common.StatSender{
			SubStatter:   f.serviceCommon.Statsd.NewSubStatter("sqs_receiver"),
			ErrorTracker: &f.serviceCommon.ErrorTracker,
		},
	}

	friendshipSQSSource := fanout.FriendshipSqsSource{
		Destination: &f.feedMux,
		SQSQueueProcessor: feedsqs.SQSQueueProcessor{
			Log: f.serviceCommon.Log,
			Sqs: sqs.New(session, awsConf...),
			Stats: &service_common.StatSender{
				SubStatter:   f.serviceCommon.Statsd.NewSubStatter("sqs_source.friendship"),
				ErrorTracker: &f.serviceCommon.ErrorTracker,
			},
			Conf: &f.configs.friendshipSqsSourceConfig.SQSQueueProcessorConfig,
			Ch:   &f.serviceCommon.Ctxlog,
		},
		FriendshipConfig: &f.configs.friendshipSqsSourceConfig,
	}

	updateFollowSQSSource := fanout.UpdateFollowSqsSource{
		Destination: &f.feedMux,
		SQSQueueProcessor: feedsqs.SQSQueueProcessor{
			Log: f.serviceCommon.Log,
			Sqs: sqs.New(session, awsConf...),
			Stats: &service_common.StatSender{
				SubStatter:   f.serviceCommon.Statsd.NewSubStatter("sqs_source.update_follow"),
				ErrorTracker: &f.serviceCommon.ErrorTracker,
			},
			Conf: &f.configs.updateFollowSqsSourceConfig.SQSQueueProcessorConfig,
			Ch:   &f.serviceCommon.Ctxlog,
		},
		UpdateFollowConfig: &f.configs.updateFollowSqsSourceConfig,
	}

	f.server = fanout.HTTPServer{
		BaseHTTPServer: service_common.BaseHTTPServer{
			Config:      &f.configs.HTTPConfig.BaseHTTPServerConfig,
			Log:         f.serviceCommon.Log,
			PanicLogger: f.serviceCommon.PanicLogger,
			Ctxlog:      &f.serviceCommon.Ctxlog,
			Dims:        &f.serviceCommon.CtxDimensions,
			ElevateKey:  f.serviceCommon.ElevateLogKey,
			Stats: &service_common.StatSender{
				SubStatter:   f.serviceCommon.Statsd.NewSubStatter("http"),
				ErrorTracker: &f.serviceCommon.ErrorTracker,
			},
			XRay:     f.serviceCommon.XRay,
			OnListen: f.onListen,
		},
		Destination:       &f.feedMux,
		CustomDestination: &f.feedMux.ActivityProcessor,
	}
	f.server.BaseHTTPServer.SetupRoutes = f.server.SetupRoutes

	f.sqsSource = fanout.SqsSource{
		Destination: &f.feedMux,
		SQSQueueProcessor: feedsqs.SQSQueueProcessor{
			Log:             f.serviceCommon.Log,
			Ch:              &f.serviceCommon.Ctxlog,
			StopWaitChannel: make(chan struct{}),
			Sqs:             sqs.New(session, awsConf...),
			Stats: &service_common.StatSender{
				SubStatter:   f.serviceCommon.Statsd.NewSubStatter("sqs_source.fanout_main"),
				ErrorTracker: &f.serviceCommon.ErrorTracker,
			},
			Conf: &f.configs.SqsSourceConfig.SQSQueueProcessorConfig,
		},
	}
	f.followsDiscovery = fanout.FollowsDiscovery{
		GraphDB: f.graphDBClient,
		Log:     f.serviceCommon.Log,
		Stats: &service_common.StatSender{
			SubStatter:   f.serviceCommon.Statsd.NewSubStatter("follows_discovery"),
			ErrorTracker: &f.serviceCommon.ErrorTracker,
		},
		Config: &f.configs.followsDiscoveryConfig,
	}
	f.friendshipDiscovery = fanout.FriendsDiscovery{
		Friendship: f.friendshipClient,
		Stats: &service_common.StatSender{
			SubStatter:   f.serviceCommon.Statsd.NewSubStatter("friendship_discovery"),
			ErrorTracker: &f.serviceCommon.ErrorTracker,
		},
		Log:    f.serviceCommon.Log,
		Config: &f.configs.friendsDiscoveryConfig,
	}

	f.feedMux = fanout.FeedDemultiplexer{
		GroupedFeedReceiver: f.destination,
		ActivityProcessor: fanout.ActivityProcessor{
			Log: f.serviceCommon.Log,
			EdgeDetectors: []fanout.ConsumerCandidateDiscovery{
				&fanout.ChannelFeed{},
				&f.followsDiscovery,
				&f.friendshipDiscovery,
			},
			BatchDiscovery: []fanout.ActivityBatchDiscovery{
				&fanout.FollowerStoryBatches{
					DuploClient: &duploClient,
					Stats: &service_common.StatSender{
						SubStatter:   f.serviceCommon.Statsd.NewSubStatter("follow_batch"),
						ErrorTracker: &f.serviceCommon.ErrorTracker,
					},
				},
			},

			MaxBatchSize: f.serviceCommon.Config.Int("masonry.story_batch.batch_size", 1000),
			Stats: &service_common.StatSender{
				SubStatter:   f.serviceCommon.Statsd.NewSubStatter("feed_demultiplexer"),
				ErrorTracker: &f.serviceCommon.ErrorTracker,
			},
		},
	}

	f.runner = service_common.ServiceRunner{
		Log: f.serviceCommon.Log,
		Services: []service_common.Service{
			&f.server, &f.serviceCommon, &f.sqsSource, &friendshipSQSSource, &updateFollowSQSSource,
		},
		SignalNotify: signal.Notify,
		SigChan:      f.sigChan,
	}
	// Cannot just pass f because f contains private members that I cannot nil check via reflection
	res := (&service_common.NilCheck{
		IgnoredPackages: []string{"aws-sdk-go", "net/http"},
	}).Check(f, &f.server, f.runner, f.sqsSource, f.feedMux, f.friendshipClient, f.friendshipDiscovery, f.destination)
	res.MustBeEmpty(os.Stderr)
}

func (f *service) main() {
	if err := f.setup(); err != nil {
		service_common.SetupLogger.Log("err", err, "Unable to load initial config")
		f.osExit(1)
		return
	}
	f.inject()

	if err := f.runner.Execute(); err != nil {
		service_common.SetupLogger.Log("err", err, "wait to end finished with an error")
		f.osExit(1)
		return
	}
	f.serviceCommon.Log.Log("Finished main")
}

func main() {
	instance.main()
}
