// +build integration

package main

import (
	"bytes"
	"fmt"
	"io"
	"net"
	"net/http"
	"os"
	"syscall"
	"testing"
	"time"

	"encoding/json"
	"io/ioutil"

	"code.justin.tv/feeds/clients/feeddataflow"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/fanout/cmd/fanout/internal/fanout"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-common/verb"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/log/fmtlogger"
	"code.justin.tv/feeds/service-common"
	"github.com/aws/aws-sdk-go/aws"
	"github.com/aws/aws-sdk-go/aws/session"
	"github.com/aws/aws-sdk-go/service/sqs"
	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
)

const exampleFriendshipBody = `{
  "Type" : "Notification",
  "MessageId" : "2bcb59d5-f77b-52bc-838a-092536ca290a",
  "TopicArn" : "arn:aws:sns:us-west-2:603200399373:friendship_production_events",
  "Message" : "{\"requester_id\":\"111754530\",\"target_id\":\"141859672\"}",
  "Timestamp" : "2017-01-11T23:27:45.821Z",
  "SignatureVersion" : "1",
  "Signature" : "Cx8oKIcq310wkBeqWd4LKOwUWM3zX9fGgvM+aBcXPHe201htAWqKOXDF5lZijKCQeyxWogH1PWCIJ7PiVrnHz3s9A2we2N9TgtyXGY50bu40pm553iIEqoa778YDsEkjANzyHvHTv85nngTZIkncqhQNMzIOez4qxjz+n8YielQLnbvXo/dcVHyi4Z5jgt0ILdQ8JfyNAoX9PaCGIbImJBM/R50MyrBAIeJ70G3D57HjqiQW/qFKBEahjXMitDFhIYGUUZcaBJ9O2wlotXLdyGiTTQDCAVlT8EmSPiRiM2cgpCQm0yX10QRTPMd/1PqntPSvG0Xei3B/eMHxAu9XPg==",
  "SigningCertURL" : "https://sns.us-west-2.amazonaws.com/SimpleNotificationService-b95095beb82e8f6a046b3aafc7f4149a.pem",
  "UnsubscribeURL" : "https://sns.us-west-2.amazonaws.com/?Action=Unsubscribe&SubscriptionArn=arn:aws:sns:us-west-2:603200399373:friendship_production_events:8a117309-ee9a-4c37-bbb1-5008f3e52169",
  "MessageAttributes" : {
    "event" : {"Type":"String","Value":"friend_request_accepted"}
  }
}`

// Activity is the feed object that comes into fanout service
type Activity struct {
	Entity entity.Entity `json:"entity"`
	Verb   verb.Verb     `json:"verb"`
	Actor  string        `json:"actor"` // not entity so I can test invalid activity
}

func addActivity(sqsClient *sqs.SQS, instance *service, activity *Activity) error {
	msgBuffer := &bytes.Buffer{}
	if err := json.NewEncoder(msgBuffer).Encode(activity); err != nil {
		return err
	}
	_, err := sqsClient.SendMessage(&sqs.SendMessageInput{
		QueueUrl:    aws.String(instance.configs.SqsSourceConfig.QueueURL.Get()),
		MessageBody: aws.String(msgBuffer.String()),
	})
	return err
}

type story struct {
	Activity *fanout.Activity
	Feed     string
	Metadata *feeddataflow.Metadata
}

func splitActivites(m *fanout.MemoryFeedStoriesReceiver) []story {
	ret := []story{}
	all := append(append(append([]*fanout.ActivityBatch{}, m.Low...), m.Mid...), m.High...)
	for _, l := range all {
		for _, activity := range l.Activities {
			for _, feed := range l.FeedIDs {
				ret = append(ret, story{
					Activity: activity,
					Feed:     feed.Feed,
					Metadata: feed.Metadata,
				})
			}
		}
	}
	return ret
}

func ShouldHaveStoryThatMatchesAssertion(actual interface{}, expected ...interface{}) string {
	oldRet := ShouldNotHaveStoryThatMatchesAssertion(actual, expected...)
	if oldRet != "" {
		return ""
	}
	return "unable to find expected story"
}

func ShouldNotHaveStoryThatMatchesAssertion(actual interface{}, expected ...interface{}) string {
	f := expected[0].(func(story) bool)
	for _, s := range actual.([]story) {
		if f(s) {
			return fmt.Sprintf("was able to find the story")
		}
	}
	return ""
}

func simulateActivity(host string, activity *Activity) (*fanout.MemoryFeedStoriesReceiver, error) {
	activityURL := host + "/private/simulate_activity"
	var buf bytes.Buffer
	if err := json.NewEncoder(&buf).Encode(activity); err != nil {
		return nil, err
	}
	req, err := http.NewRequest("POST", activityURL, &buf)
	if err != nil {
		return nil, err
	}
	resp, err := http.DefaultClient.Do(req)
	if err != nil {
		return nil, err
	}
	rest, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return nil, err
	}
	if resp.StatusCode != http.StatusOK {
		return nil, errors.Errorf("invalid status %d %s", resp.StatusCode, string(rest))
	}
	var ret fanout.MemoryFeedStoriesReceiver
	if err := json.Unmarshal(rest, &ret); err != nil {
		return nil, errors.Wrap(err, "invalid body "+string(rest))
	}
	return &ret, nil
}

func TestFanoutService(t *testing.T) {
	host, instance, logWatcher, cleanShutdown := startServer(t)
	if host == "" {
		t.Error("Unable to setup testing server")
		return
	}

	Convey("With "+host, t, func(c C) {
		ts := &testSetup{host: host}
		So(ts.Setup(), ShouldBeNil)
		c.Reset(ts.cancelFunc)
		awsSession, awsConf := service_common.CreateAWSSession(instance.serviceCommon.Config)
		ctx := context.Background()
		sqsClient := sqs.New(awsSession, awsConf...)
		Convey("Should log invalid messages", func() {
			shouldComplainAboutActor := logWatcher.FindFirstSubstr("INVALID_ACTOR_ID")
			So(addActivity(sqsClient, instance, &Activity{
				Entity: entity.New("testing", "5"),
				Verb:   verb.Create,
				Actor:  "INVALID_ACTOR_ID",
			}), ShouldBeNil)
			ctx2, cancel := context.WithTimeout(ctx, time.Second*5)
			defer cancel()
			So(shouldComplainAboutActor.Block(ctx2), ShouldBeTrue)
		})
		Convey("should silently accept messages from people with no friends", func() {
			shouldComplainAboutActor := logWatcher.FindFirstSubstr("Unable to process SQS message")
			So(addActivity(sqsClient, instance, &Activity{
				Entity: entity.New("testing", "4"),
				Verb:   verb.Create,
				Actor:  entity.New(entity.NamespaceUser, "1012312312").Encode(),
			}), ShouldBeNil)
			ctx2, cancel := context.WithTimeout(ctx, time.Second*2)
			defer cancel()
			So(shouldComplainAboutActor.Block(ctx2), ShouldBeFalse)
		})

		Convey("sending a post activity", func() {
			actorMainUser := "327184041"
			activity := Activity{
				Entity: entity.New(entity.NamespacePost, "123"),
				Verb:   verb.Create,
				Actor:  entity.NamespaceUser + ":" + actorMainUser,
			}
			resp, err := simulateActivity(host, &activity)
			So(err, ShouldBeNil)
			allActivity := splitActivites(resp)
			So(allActivity, ShouldNotBeNil)

			// Your own channel feed
			So(allActivity, ShouldHaveStoryThatMatchesAssertion, func(s story) bool {
				return s.Feed == "c:"+actorMainUser
			})

			// Your own news feed
			So(allActivity, ShouldNotHaveStoryThatMatchesAssertion, func(s story) bool {
				return s.Feed == "n:"+actorMainUser
			})
		})

		Convey("sending a follow activity should not fan out", func() {
			actorMainUser := "327184041"
			activity := Activity{
				Entity: entity.New(entity.NamespaceFollow, "123"),
				Verb:   verb.Create,
				Actor:  entity.NamespaceUser + ":" + actorMainUser,
			}
			resp, err := simulateActivity(host, &activity)
			So(err, ShouldBeNil)
			allActivity := splitActivites(resp)
			So(allActivity, ShouldNotBeNil)

			// Your own channel feed
			So(allActivity, ShouldNotHaveStoryThatMatchesAssertion, func(s story) bool {
				return s.Activity.Entity.Namespace() == entity.NamespaceFollow
			})

			// Your own news feed
			So(allActivity, ShouldNotHaveStoryThatMatchesAssertion, func(s story) bool {
				return s.Activity.Entity.Namespace() == entity.NamespaceFollow
			})
		})

		Convey("sending a friendship activity should not fan out", func() {
			actorMainUser := "327184041"
			activity := Activity{
				Entity: entity.New(entity.NamespaceFollow, "123"),
				Verb:   verb.Create,
				Actor:  entity.NamespaceUser + ":" + actorMainUser,
			}
			resp, err := simulateActivity(host, &activity)
			So(err, ShouldBeNil)
			allActivity := splitActivites(resp)
			So(allActivity, ShouldNotBeNil)

			// Your own channel feed
			So(allActivity, ShouldNotHaveStoryThatMatchesAssertion, func(s story) bool {
				return s.Activity.Entity.Namespace() == entity.NamespaceFriend
			})

			// Your own news feed
			So(allActivity, ShouldNotHaveStoryThatMatchesAssertion, func(s story) bool {
				return s.Activity.Entity.Namespace() == entity.NamespaceFriend
			})
		})
	})

	cleanShutdown(time.Second * 15)
}

type testSetup struct {
	ctx        context.Context
	cancelFunc func()
	client     *http.Client
	host       string
}

func (t *testSetup) Setup() error {
	t.ctx, t.cancelFunc = context.WithTimeout(context.Background(), time.Second*15)
	t.client = &http.Client{}
	return nil
}

func addMapValues(m *distconf.InMemory, vals map[string][]byte) error {
	for k, v := range vals {
		if err := m.Write(k, v); err != nil {
			return err
		}
	}
	return nil
}

func queuePrefix() string {
	loginUser := os.Getenv("USER")
	if loginUser == "" {
		loginUser = "unknown"
	}
	return "fanout_testing_" + loginUser + "_"
}

type panicPanic struct{}

func (p panicPanic) OnPanic(pnc interface{}) {
	panic(pnc)
}

func startServer(t *testing.T) (string, *service, *service_common.LineOutputWatcher, func(time.Duration)) {
	logWatcher := &service_common.LineOutputWatcher{
		Out: ioutil.Discard,
	}
	localConf := &distconf.InMemory{}
	err := addMapValues(localConf, map[string][]byte{
		"fanout.listen_addr":                             []byte(":0"),
		"rollbar.access_token":                           []byte(""),
		"statsd.hostport":                                []byte(""),
		"friendship-activity.sqssource.draining_threads": []byte("1"),
		"fanout.sqssource.draining_threads":              []byte("1"),
		"debug.addr":                                     []byte(":0"),
		"logging.to_stdout":                              []byte("false"),
	})
	if err != nil {
		t.Error(err)
		return "", nil, nil, nil
	}

	started := make(chan string)
	finished := make(chan struct{})
	signalToClose := make(chan os.Signal)
	exitCalled := make(chan struct{})
	queuesToRemove := make([]string, 0, 3)
	var sqsClient *sqs.SQS
	elevateKey := "hi"
	logout := log.MultiLogger([]log.Logger{t, fmtlogger.NewLogfmtLogger(logWatcher, log.Discard)})

	thisInstance := service{
		osExit: func(i int) {
			if i != 0 {
				t.Error("Invalid osExit status code", i)
			}
			close(exitCalled)
		},
		awsSetup: makeTestingAWSSetup(&sqsClient, t, &queuesToRemove, localConf),

		serviceCommon: service_common.ServiceCommon{
			ConfigCommon: service_common.ConfigCommon{
				Team:          teamName,
				Service:       serviceName,
				CustomReaders: []distconf.Reader{localConf},
				BaseDirectory: "../../",
				OsGetenv:      os.Getenv,
				OsHostname:    os.Hostname,
			},
			CodeVersion: CodeVersion,
			Log: &log.ElevatedLog{
				ElevateKey: elevateKey,
				NormalLog: log.ContextLogger{
					Logger: logout,
				},
				DebugLog: log.ContextLogger{
					Logger: log.Discard,
				},
				LogToDebug: func(vals ...interface{}) bool {
					return false
				},
			},
			PanicLogger:    panicPanic{},
			SfxSetupConfig: sfxStastdConfig(),
		},
		sigChan: signalToClose,
		onListen: func(listeningAddr net.Addr) {
			started <- fmt.Sprintf("http://localhost:%d", listeningAddr.(*net.TCPAddr).Port)
		},
	}
	thisInstance.serviceCommon.Log.NormalLog.Dims = &thisInstance.serviceCommon.CtxDimensions
	thisInstance.serviceCommon.Log.DebugLog.Dims = &thisInstance.serviceCommon.CtxDimensions

	go func() {
		thisInstance.main()
		close(finished)
	}()

	var addressForIntegrationTests string
	select {
	case <-exitCalled:
		return "", nil, nil, nil
	case addressForIntegrationTests = <-started:
	case <-time.After(time.Second * 15):
		t.Error("Took to long to start service")
		return "", nil, nil, nil
	}

	c := http.Client{}
	resp, err := c.Get(addressForIntegrationTests + "/debug/health")
	if err != nil {
		t.Error(err)
		return "", nil, nil, nil
	}
	if resp.StatusCode != http.StatusOK {
		t.Error("Invalid status code ", resp.StatusCode)
		buf := &bytes.Buffer{}
		_, err := io.Copy(buf, resp.Body)
		t.Error(buf.String(), err)
		return "", nil, nil, nil
	}
	onFinish := makeTestingCloseFunc(signalToClose, sqsClient, t, queuesToRemove, finished)
	return addressForIntegrationTests, &thisInstance, logWatcher, onFinish
}

func makeTestingAWSSetup(sqsClient **sqs.SQS, t *testing.T, queuesToRemove *[]string, localConf *distconf.InMemory) func(*session.Session, []*aws.Config) error {
	dconfToQueue := map[string]string{
		"fanout.sqssource.queue_url":      queuePrefix() + "input",
		"masonry.low.sqssource.queue_url": queuePrefix() + "masonry_low",
		"masonry.mid.sqssource.queue_url": queuePrefix() + "masonry_mid",
	}
	return func(session *session.Session, awsConf []*aws.Config) error {
		*sqsClient = sqs.New(session, awsConf...)
		for dconfKey, queueName := range dconfToQueue {
			output, err := (*sqsClient).CreateQueue(&sqs.CreateQueueInput{
				QueueName: &queueName,
			})
			if err != nil {
				return err
			}
			t.Log("for key", dconfKey, "queue URL is ", *output.QueueUrl)
			*queuesToRemove = append(*queuesToRemove, *output.QueueUrl)
			if err := localConf.Write(dconfKey, []byte(*output.QueueUrl)); err != nil {
				return err
			}
		}
		return nil
	}
}

func makeTestingCloseFunc(signalToClose chan os.Signal, sqsClient *sqs.SQS, t *testing.T, queuesToRemove []string, finished chan struct{}) func(time.Duration) {
	return func(timeToWait time.Duration) {
		signalToClose <- syscall.SIGTERM
		defer func() {
			if sqsClient != nil {
				t.Log("Staring to clean up SQS queues")
				for _, queueUrl := range queuesToRemove {
					_, err := sqsClient.DeleteQueue(&sqs.DeleteQueueInput{
						QueueUrl: &queueUrl,
					})
					if err != nil {
						t.Log("Unable to remove a SQS queue at the end", err)
					}
					t.Log("Removed queue with URL ", queueUrl)
				}
				t.Log("finished cleaning up SQS queues")
			}
		}()
		select {
		case <-finished:
			return
		case <-time.After(timeToWait):
			t.Error("Timed out waiting for server to end")
		}
	}
}
