locals {
  owner   = "jlindamo@twitch.tv"
  team    = "feeds"
  region  = "us-west-2"
  service = "fanout"

  account_ids = {
    integration = "724951484461"
    staging     = "724951484461"
    production  = "914569885343"
  }

  profiles = {
    integration = "twitch-feed-dev"
    staging     = "twitch-feed-dev"
    production  = "twitch-feed-aws"
  }

  min_counts = {
    integration = "0"
    staging     = "0"
    production  = "0"
  }

  desired_counts = {
    integration = "0"
    staging     = "0"
    production  = "0"
  }

  max_counts = {
    integration = "0"
    staging     = "0"
    production  = "0"
  }

  auto_promotes = {
    integration = "true"
    staging     = "true"
    production  = "false"
  }

  promote_froms = {
    integration = "latest"
    staging     = "integration"
    production  = "canary"
  }

  pagerduty_integration_urls = {
    integration = ""
    staging     = ""
    production  = ""
  }

  friendship_sns_arns = {
    integration = "arn:aws:sns:us-west-2:603200399373:friendship_staging_events"
    staging     = "arn:aws:sns:us-west-2:603200399373:friendship_staging_events"
    production  = "arn:aws:sns:us-west-2:603200399373:friendship_production_events"
  }

  update_follow_sns_arns = {
    integration = "arn:aws:sns:us-west-2:724951484461:following-service_staging_updatefollow_notifications"
    staging     = "arn:aws:sns:us-west-2:724951484461:following-service_staging_updatefollow_notifications"
    production  = "arn:aws:sns:us-west-2:914569885343:following-service_production_updatefollow_notifications"
  }

  redis_instance_types = {
    integration = "cache.r3.xlarge"
    staging     = "cache.r3.xlarge"
    production  = "cache.r3.4xlarge"
  }

  redis_cache_nodes_counts = {
    integration = "1"
    staging     = "1"
    production  = "1"
  }

  pushy_comment_sqs_arns = {
    integration = ""
    staging     = "arn:aws:sqs:us-west-2:603200399373:pushy_darklaunch_feedcomment"
    production  = "arn:aws:sqs:us-west-2:603200399373:pushy_production_feedcomment"
  }

  pagerduty_integration_url = "${lookup(local.pagerduty_integration_urls, terraform.workspace)}"
  pushy_comment_sqs_arn     = "${lookup(local.pushy_comment_sqs_arns, terraform.workspace)}"
  friendship_sns_arn        = "${lookup(local.friendship_sns_arns, terraform.workspace)}"
  redis_cache_nodes         = "${lookup(local.redis_cache_nodes_counts, terraform.workspace)}"
  redis_instance_type       = "${lookup(local.redis_instance_types, terraform.workspace)}"
  update_follow_sns_arn     = "${lookup(local.update_follow_sns_arns, terraform.workspace)}"

  promote_from  = "${lookup(local.promote_froms, terraform.workspace)}"
  auto_promote  = "${lookup(local.auto_promotes, terraform.workspace)}"
  min_count     = "${lookup(local.min_counts, terraform.workspace)}"
  max_count     = "${lookup(local.max_counts, terraform.workspace)}"
  desired_count = "${lookup(local.desired_counts, terraform.workspace)}"
  profile       = "${lookup(local.profiles, terraform.workspace)}"
}

provider "aws" {
  profile             = "${local.profile}"
  allowed_account_ids = ["${lookup(local.account_ids, terraform.workspace)}"]
  region              = "${local.region}"
}

terraform {
  backend "s3" {
    bucket  = "twitch-feed-aws"
    key     = "feeds/fanout-us-west-2.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

provider "jenkins" {
  url                = "https://jenkins.internal.justin.tv"
  shared_credentials = "${file(pathexpand("~/.jenkins_auth"))}"
}

provider "consul" {
  address    = "consul.internal.justin.tv"
  datacenter = "us-west2"
}

data "terraform_remote_state" "account" {
  backend = "s3"

  config {
    bucket  = "${local.profile}"
    key     = "tfstate/feed/terraform/accounts/${local.profile}"
    region  = "us-west-2"
    profile = "${local.profile}"
  }
}

data "terraform_remote_state" "account_core" {
  backend   = "s3"
  workspace = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "feeds/core-account.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

data "terraform_remote_state" "region_core" {
  backend   = "s3"
  workspace = "${terraform.workspace}"

  config {
    bucket  = "twitch-feed-aws"
    key     = "feeds/core-region-${local.region}.tfstate"
    region  = "us-west-2"
    profile = "twitch-feed-aws"
  }
}

resource "aws_sqs_queue" "input_deadletter" {
  name = "${local.service}-${terraform.workspace}-input_deadletter"

  lifecycle {
    prevent_destroy = true
  }

  tags {
    Service     = "${local.service}"
    Environment = "${terraform.workspace}"
    Team        = "${local.team}"
  }
}

resource "aws_sqs_queue" "input_queue" {
  name                       = "${local.service}-${terraform.workspace}-input"
  visibility_timeout_seconds = 900
  redrive_policy             = "{\"deadLetterTargetArn\":\"${aws_sqs_queue.input_deadletter.arn}\",\"maxReceiveCount\":3}"

  tags {
    Service     = "${local.service}"
    Environment = "${terraform.workspace}"
    Team        = "${local.team}"
  }
}

resource "aws_sqs_queue" "output_deadletter" {
  name = "${local.service}-${terraform.workspace}-output_deadletter"

  lifecycle {
    prevent_destroy = true
  }

  tags {
    Service     = "${local.service}"
    Environment = "${terraform.workspace}"
    Team        = "${local.team}"
  }
}

resource "aws_sqs_queue" "output_queue" {
  name           = "${local.service}-${terraform.workspace}-output"
  redrive_policy = "{\"deadLetterTargetArn\":\"${aws_sqs_queue.output_deadletter.arn}\",\"maxReceiveCount\":3}"

  lifecycle {
    prevent_destroy = true
  }

  tags {
    Service     = "${local.service}"
    Environment = "${terraform.workspace}"
    Team        = "${local.team}"
  }
}

# Low pri q
resource "aws_sqs_queue" "output_lowpri_queue" {
  name           = "${local.service}-${terraform.workspace}-output-lowpri"
  redrive_policy = "{\"deadLetterTargetArn\":\"${aws_sqs_queue.output_deadletter.arn}\",\"maxReceiveCount\":3}"

  lifecycle {
    prevent_destroy = true
  }

  tags {
    Service     = "${local.service}"
    Environment = "${terraform.workspace}"
    Team        = "${local.team}"
  }
}

# High pri q
resource "aws_sqs_queue" "output_highpri_queue" {
  name           = "${local.service}-${terraform.workspace}-output-highpri"
  redrive_policy = "{\"deadLetterTargetArn\":\"${aws_sqs_queue.output_deadletter.arn}\",\"maxReceiveCount\":3}"

  lifecycle {
    prevent_destroy = true
  }

  tags {
    Service     = "${local.service}"
    Environment = "${terraform.workspace}"
    Team        = "${local.team}"
  }
}

# Friendship SNS messages
resource "aws_sqs_queue" "friendship_requests" {
  name = "${local.service}-${terraform.workspace}-friendship-changes"

  lifecycle {
    prevent_destroy = true
  }

  tags {
    Service     = "${local.service}"
    Environment = "${terraform.workspace}"
    Team        = "${local.team}"
  }
}

# This only works if our account already has permission for this SNS topic.
# This is a two step process.
# (1) This change adds the subscription to the SNS topic to send messages to
# our SQS queue
resource "aws_sns_topic_subscription" "user_updates_sqs_target" {
  topic_arn = "${local.friendship_sns_arn}"
  protocol  = "sqs"
  endpoint  = "${aws_sqs_queue.friendship_requests.arn}"
}

# (2) This change adds permissions to the SQS queue that allow the SNS topic to
# SendMessages to it.
resource "aws_sqs_queue_policy" "friendship-sns-perm" {
  queue_url = "${aws_sqs_queue.friendship_requests.id}"

  policy = <<POLICY
{
  "Version": "2012-10-17",
  "Id": "sqspolicy",
  "Statement": [
    {
      "Sid": "First",
      "Effect": "Allow",
      "Principal": "*",
      "Action": "SQS:SendMessage",
      "Resource": "${aws_sqs_queue.friendship_requests.arn}",
      "Condition": {
        "ArnEquals": {
          "aws:SourceArn": "${local.friendship_sns_arn}"
        }
      }
    }
  ]
}
POLICY
}

# Following SNS messages
resource "aws_sqs_queue" "update_follow" {
  name = "${local.service}-${terraform.workspace}-update-follow"

  lifecycle {
    prevent_destroy = true
  }

  tags {
    Service     = "${local.service}"
    Environment = "${terraform.workspace}"
    Team        = "${local.team}"
  }
}

# Add the subscription to the SNS topic
resource "aws_sns_topic_subscription" "update_follow_sqs_target" {
  topic_arn = "${local.update_follow_sns_arn}"
  protocol  = "sqs"
  endpoint  = "${aws_sqs_queue.update_follow.arn}"
}

# Add permissions to the SQS queue
resource "aws_sqs_queue_policy" "update-follow-sns-perm" {
  queue_url = "${aws_sqs_queue.update_follow.id}"

  policy = <<POLICY
{
  "Version": "2012-10-17",
  "Id": "sqspolicy",
  "Statement": [
    {
      "Sid": "First",
      "Effect": "Allow",
      "Principal": "*",
      "Action": "SQS:SendMessage",
      "Resource": "${aws_sqs_queue.update_follow.arn}",
      "Condition": {
        "ArnEquals": {
          "aws:SourceArn": "${local.update_follow_sns_arn}"
        }
      }
    }
  ]
}
POLICY
}

resource "aws_sns_topic" "comment_notfication_topic" {
  name = "comment_notification_${terraform.workspace}_topic"

  lifecycle {
    prevent_destroy = true
  }
}

resource "aws_sns_topic_policy" "comment_notfication_topic_policy" {
  arn = "${aws_sns_topic.comment_notfication_topic.arn}"

  policy = <<EOF
{
  "Version": "2012-10-17",
  "Id": "${local.pushy_comment_sqs_arn}/policy",
  "Statement":[{
    "Sid": "1211239f-dd31-4daa-9e89-ed0f8f14d207",
    "Effect": "Allow",
    "Principal": {"AWS":"*"},
    "Action": "SNS:Subscribe",
    "Resource": "${aws_sns_topic.comment_notfication_topic.arn}",
    "Condition": {
      "ArnEquals": {
        "aws:SourceArn": "${local.pushy_comment_sqs_arn}"
      }
    }
  }]
}
EOF
}

module "fanout" {
  source           = "../../modules/cluster"
  environment      = "${terraform.workspace}"
  target_group_arn = "${module.alb.target_group_arn}"
  cluster_name     = "${data.terraform_remote_state.region_core.common_cluster_name}"

  min_size                  = "${local.min_count}"
  desired_count             = "${local.desired_count}"
  max_size                  = "${local.max_count}"
  service_iam_role          = "${data.terraform_remote_state.account_core.service_iam_role}"
  task_autoscale_iam_role   = "${data.terraform_remote_state.account_core.task_autoscale_iam_role}"
  profile                   = "${local.profile}"
  promote_from              = "${local.promote_from}"
  auto_promote              = "${local.auto_promote}"
  deploy_aws_role           = "${data.terraform_remote_state.account_core.build_role}"
  container_loggroup_region = "${data.terraform_remote_state.region_core.container_loggroup_region}"
  container_loggroup        = "${data.terraform_remote_state.region_core.container_loggroup}"
  aws_creds                 = "${data.terraform_remote_state.account_core.jenkins_build_secret}"
  testing_assume_role       = "${data.terraform_remote_state.account_core.build_user_arn}"
}

module "permissions" {
  source      = "../../modules/permissions"
  environment = "${terraform.workspace}"
  iam_id      = "${module.fanout.iam_id}"

  friendship_requests_arn  = "${aws_sqs_queue.friendship_requests.arn}"
  output_lowpri_queue_arn  = "${aws_sqs_queue.output_lowpri_queue.arn}"
  output_queue_arn         = "${aws_sqs_queue.output_queue.arn}"
  input_queue_arn          = "${aws_sqs_queue.input_queue.arn}"
  update_follow_arn        = "${aws_sqs_queue.update_follow.arn}"
  output_highpri_queue_arn = "${aws_sqs_queue.output_highpri_queue.arn}"
}

module "alb" {
  source            = "git::git+ssh://git@git-aws.internal.justin.tv/twitch/terraform//alb?ref=v0.8.20"
  owner             = "${local.owner}"
  environment       = "${terraform.workspace}"
  access_log_bucket = "${local.profile}-logs"
  team              = "${local.team}"
  service           = "${local.service}"
  subnets           = "${data.terraform_remote_state.account.private_subnets}"
  security_groups   = "${data.terraform_remote_state.account.twitch_subnets_sg}"
  vpc_id            = "${data.terraform_remote_state.account.vpc_id}"
}

output target_group_arn {
  value = "${module.alb.target_group_arn}"
}

output task_role_arn {
  value = "${module.fanout.task_role_arn}"
}

output friendship_requests_arn {
  value = "${aws_sqs_queue.friendship_requests.arn}"
}

output output_lowpri_queue_arn {
  value = "${aws_sqs_queue.output_lowpri_queue.arn}"
}

output output_queue_arn {
  value = "${aws_sqs_queue.output_queue.arn}"
}

output input_queue_arn {
  value = "${aws_sqs_queue.input_queue.arn}"
}

output update_follow_arn {
  value = "${aws_sqs_queue.update_follow.arn}"
}

output output_highpri_queue_arn {
  value = "${aws_sqs_queue.output_highpri_queue.arn}"
}
