package api

import (
	"net/http"

	"github.com/afex/hystrix-go/hystrix"
	"github.com/cactus/go-statsd-client/statsd"
	"goji.io"
	"goji.io/pat"
	"golang.org/x/net/context"

	"code.justin.tv/chat/friendship/app/backend"
	"code.justin.tv/chat/friendship/clients"
	"code.justin.tv/chat/friendship/rpc/friends"
	"code.justin.tv/chat/golibs/errx"
	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/golibs/gojiplus/decorators"
	"code.justin.tv/chat/golibs/gojiplus/middleware"
	"code.justin.tv/chat/golibs/logx"
)

// Server contains the essential parts for the Friendship server - stats, backend, errorlogger, and httpHandler.
type Server struct {
	http.Handler
	stats   statsd.Statter
	backend backend.Backender
}

// NewServer creates a new instance of a Server.
func NewServer(b backend.Backender, stats statsd.Statter) (*Server, error) {
	s := &Server{
		stats:   stats,
		backend: b,
	}

	root := goji.NewMux()
	root.UseC(middleware.ClientRowID)
	root.Handle(pat.Get("/debug/health"), gojiplus.HealthCheck())

	// Kraken endpoints (/users/:user_id/*)
	usersV0 := goji.SubMux()
	root.HandleC(pat.New("/users/:user_id/*"), usersV0)
	usersV0.Use(middleware.EnhancedResponseWriter)

	usersV0.HandleFuncC(pat.Get("/friends/requests"), s.createHandler("get_incoming_requests_v0", s.getIncomingRequestsV0))
	usersV0.HandleFuncC(pat.Put("/friends/requests/:friend_id"), s.createHandler("add_friend_v0", s.addFriendV0))
	usersV0.HandleFuncC(pat.Delete("/friends/requests/:friend_id"), s.createHandler("reject_request_v0", s.rejectRequestV0))

	usersV0.HandleFuncC(pat.Get("/friends/notifications"), s.createHandler("get_notifications_v0", s.getNotificationsV0))
	usersV0.HandleFuncC(pat.Delete("/friends/notifications"), s.createHandler("reset_notification_v0", s.resetNotificationsV0))

	usersV0.HandleFuncC(pat.Get("/friends/recommended"), s.createHandler("get_recommended_v0", s.getRecommendedV0))
	usersV0.HandleFuncC(pat.Delete("/friends/recommended/:target_id"), s.createHandler("dismiss_recommended_v0", s.dismissRecommendedV0))

	usersV0.HandleFuncC(pat.Get("/friends"), s.createHandler("get_friends_v0", s.listFriendsV0))
	usersV0.HandleFuncC(pat.Delete("/friends"), s.createHandler("delete_user_v0", s.deleteUserV0))
	// These routes are placed near the bottom to not conflict with /requests, /notifications, etc.
	usersV0.HandleFuncC(pat.Get("/friends/:friend_id"), s.createHandler("get_friendship_v0", s.getFriendshipV0))
	usersV0.HandleFuncC(pat.Put("/friends/:friend_id"), s.createHandler("accept_request_v0", s.acceptRequestV0))
	usersV0.HandleFuncC(pat.Delete("/friends/:friend_id"), s.createHandler("delete_friend_v0", s.deleteFriendshipV0))

	// v1 internal endpoints (/v1/users/:user_id/*)
	usersV1 := goji.SubMux()
	root.HandleC(pat.New("/v1/users/:user_id/*"), usersV1)
	usersV1.Use(middleware.EnhancedResponseWriter)
	usersV1.UseC(middleware.TwitchRepository)

	usersV1.HandleFuncC(pat.Get("/friends/requests"), s.createHandler("get_incoming_requests_v1", s.getIncomingRequestsV1))
	usersV1.HandleFuncC(pat.Get("/friends/outgoing_requests"), s.createHandler("get_outgoing_requests_v1", s.getOutgoingRequestsV1))
	usersV1.HandleFuncC(pat.Delete("/friends/:friend_id/requests"), s.createHandler("reject_request_v1", s.rejectRequestV0))

	usersV1.HandleFuncC(pat.Get("/friends/notifications/count"), s.createHandler("get_notifications_v1", s.getNotificationsV0))
	usersV1.HandleFuncC(pat.Delete("/friends/notifications"), s.createHandler("reset_notification_v1", s.resetNotificationsV0))

	usersV1.HandleFuncC(pat.Get("/friends/recommended"), s.createHandler("get_recommended_v1", s.getRecommendedV1))

	usersV1.HandleFuncC(pat.Get("/friends"), s.createHandler("get_friends_v1", s.listFriendsV0))
	usersV1.HandleFuncC(pat.Get("/friends/:target_id/friend"), s.createHandler("is_friend_v1", s.isFriendV0))
	// These routes are placed near the bottom to not conflict with /requests, /notifications, etc.
	usersV1.HandleFuncC(pat.Get("/friends/:friend_id"), s.createHandler("get_friendship_v1", s.getFriendshipV0))
	usersV1.HandleFuncC(pat.Post("/friends/:friend_id"), s.createHandler("add_friend_v1", s.addFriendV0))
	usersV1.HandleFuncC(pat.Put("/friends"), s.createHandler("bulk_add_friends_v1", s.bulkAddFriendsV1))
	usersV1.HandleFuncC(pat.Put("/friends/:friend_id"), s.createHandler("accept_request_v1", s.acceptRequestV0))
	usersV1.HandleFuncC(pat.Delete("/friends/:friend_id"), s.createHandler("delete_friend_v1", s.deleteFriendshipV0))

	usersV1.HandleFuncC(pat.Get("/friends/:target_id/familiar"), s.createHandler("is_familiar_v1", s.isFamiliarV1))

	protobufServer := friends.NewFriendsServer(s, nil, nil)
	root.Handle(pat.Post(friends.FriendsPathPrefix+"*"), protobufServer)

	s.Handler = root
	return s, nil
}

// withHystrixFn wraps an endpoint with hystrix timeout and concurrency limits.
// If an endpoint exceeds a hystrix limit, the endpoint immediately responds with 429.
func withHystrixFn(hystrixName string) gojiplus.Decorator {
	return func(h goji.HandlerFunc) goji.HandlerFunc {
		return func(ctx context.Context, w http.ResponseWriter, r *http.Request) {
			err := hystrix.Do(hystrixName, func() error {
				h(ctx, w, r)
				if gw, ok := w.(*gojiplus.ResponseWriter); ok && gojiplus.IsServerError(gw.Status()) {
					return errx.New("received 5xx")
				}
				return nil
			}, nil)
			if err != nil {
				gojiplus.ServeError(ctx, w, r, err, http.StatusServiceUnavailable)
			}
		}
	}
}

func (s *Server) createHandler(statName string, h goji.HandlerFunc) goji.HandlerFunc {
	d := decorators.Join(
		decorators.TimingFn(s.stats, "endpoint."+statName),
		decorators.DynamoDBCapacityFn(s.stats, "endpoint."+statName),
	)

	// Wrap with a hystrix decorator if a hystrix config exists with the same statName.
	if clients.HasHystrixCommand(statName) {
		d = decorators.Join(withHystrixFn(statName), d)
	}

	decorated := d(h)

	return func(ctx context.Context, w http.ResponseWriter, r *http.Request) {
		defer logx.RecoverAndLog()

		ctx = logx.WithFields(ctx, logx.Fields{
			"method": r.Method,
			"ip":     gojiplus.GetIPFromRequest(r),
			"url":    r.URL.Path,
		})
		decorated(ctx, w, r)
	}
}
