package api

import (
	"context"

	"github.com/golang/protobuf/ptypes"

	"code.justin.tv/chat/friendship/client"
	"code.justin.tv/chat/friendship/rpc/friends"
	"code.justin.tv/chat/golibs/errx"
)

func (s Server) GetFriendships(
	ctx context.Context,
	req *friends.GetFriendshipsReq,
) (*friends.GetFriendshipsResp, error) {

	var sortDirection string
	switch req.SortDirection {
	case friends.SortDirection_ASC:
		sortDirection = friendship.AscendingDir
	case friends.SortDirection_DESC:
		sortDirection = friendship.DescendingDir
	case friends.SortDirection_DEFAULT:
		sortDirection = friendship.DescendingDir
	default:
		return nil, errx.New("invalid sort direction")
	}

	resp, err := s.backend.GetFriends(ctx, req.UserId, int(req.Limit), req.Cursor, sortDirection)
	if err != nil {
		return nil, err
	}

	var friendships []*friends.Friendship
	for _, friendship := range resp.Friends.Associations {
		createdAt, err := ptypes.TimestampProto(friendship.Time)
		if err != nil {
			return nil, err
		}

		friendships = append(friendships, &friends.Friendship{
			UserId:    friendship.Entity.ID,
			CreatedAt: createdAt,
			Cursor:    friendship.Cursor,
		})
	}

	return &friends.GetFriendshipsResp{
		Friendships: friendships,
		Total:       int64(resp.Total),
	}, nil
}
