package api

import (
	"context"

	"github.com/golang/protobuf/ptypes"

	"code.justin.tv/chat/friendship/app/backend"
	"code.justin.tv/chat/friendship/app/internal/cursor"
	"code.justin.tv/chat/friendship/client"
	"code.justin.tv/chat/friendship/rpc/friends"
	"code.justin.tv/chat/golibs/errx"
)

func (s Server) GetIncomingFriendRequests(
	ctx context.Context,
	req *friends.GetIncomingFriendRequestsReq,
) (*friends.GetIncomingFriendRequestsResp, error) {

	cursorRelationship := backend.RequestedByUser
	cursorToken := ""
	if c := req.Cursor; c != "" {
		cursorObj, err := cursor.Parse(c)
		if err != nil {
			return nil, errx.New("couldn't parse cursor")
		}
		cursorRelationship = cursorObj.Relationship
		cursorToken = cursorObj.Cursor
	}

	var sort string
	switch req.SortDirection {
	case friends.SortDirection_ASC:
		sort = friendship.AscendingDir
	case friends.SortDirection_DESC:
		sort = friendship.DescendingDir
	case friends.SortDirection_DEFAULT:
		sort = friendship.DescendingDir
	default:
		return nil, errx.New("invalid sort direction")
	}

	resp, err := s.backend.GetIncomingRequestsV1(ctx, req.UserId, cursorRelationship, cursorToken, sort, int(req.Limit))
	if err != nil {
		return nil, err
	}

	var friendRequests []*friends.IncomingFriendRequest
	for _, friendRequest := range resp.Requests {
		createdAt, err := ptypes.TimestampProto(friendRequest.RequestedAt)
		if err != nil {
			return nil, err
		}

		friendRequests = append(friendRequests, &friends.IncomingFriendRequest{
			UserId:    friendRequest.UserID,
			CreatedAt: createdAt,
			Cursor:    friendRequest.Cursor,
		})
	}

	notificationCount, err := s.backend.GetNotifications(ctx, req.UserId)
	if err != nil {
		return nil, err
	}

	recentResp, err := s.getMostRecentIncomingRequest(ctx, req.UserId)
	if err != nil {
		return nil, err
	}

	var mostRecent *friends.IncomingFriendRequest
	if recentResp != nil {
		createdAt, err := ptypes.TimestampProto(recentResp.RequestedAt)
		if err != nil {
			return nil, err
		}

		mostRecent = &friends.IncomingFriendRequest{
			UserId:    recentResp.UserID,
			CreatedAt: createdAt,
			Cursor:    recentResp.Cursor,
		}
	}

	return &friends.GetIncomingFriendRequestsResp{
		IncomingFriendRequests: friendRequests,
		Total:       int64(resp.Total),
		TotalUnread: int64(notificationCount),
		MostRecent:  mostRecent,
	}, nil
}

func (s Server) getMostRecentIncomingRequest(ctx context.Context, userID string) (*friendship.IncomingRequest, error) {
	mostRecentStrangerResult, err := s.backend.GetIncomingRequestsV1(ctx, userID, backend.RequestedByStranger, "", friendship.DescendingDir, 1)
	if err != nil {
		return nil, err
	}

	mostRecentUserResult, err := s.backend.GetIncomingRequestsV1(ctx, userID, backend.RequestedByUser, "", friendship.DescendingDir, 1)
	if err != nil {
		return nil, err
	}

	strangerRequests := mostRecentStrangerResult.Requests
	userRequests := mostRecentUserResult.Requests

	if len(strangerRequests) == 0 {
		if len(userRequests) == 0 {
			return nil, nil
		}
		return &userRequests[0], nil
	}
	if len(userRequests) == 0 {
		return &strangerRequests[0], nil
	}
	if strangerRequests[0].RequestedAt.Unix() < userRequests[0].RequestedAt.Unix() {
		return &strangerRequests[0], nil
	}
	return &userRequests[0], nil
}
