package api

import (
	"net/http"

	"golang.org/x/net/context"

	"code.justin.tv/chat/friendship/app/api/responses"
	"code.justin.tv/chat/friendship/app/api/utils"
	"code.justin.tv/chat/friendship/app/backend"
	"code.justin.tv/chat/friendship/app/internal"
	"code.justin.tv/chat/friendship/app/internal/cursor"
	friendship "code.justin.tv/chat/friendship/client"
	"code.justin.tv/chat/golibs/gojiplus"
)

const defaultIncomingRequestsLimit = 25

// getIncomingRequestsV1 retrieves incoming friends requests for a user.
//
// Deprecated: Use getIncomingRequestsV2 instead.
func (s *Server) getIncomingRequestsV1(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	required, err := gojiplus.FetchRequiredURLParams(ctx, "user_id")
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, http.StatusBadRequest)
		return
	}
	userID := required["user_id"]

	limit, err := requestLimit(r, defaultIncomingRequestsLimit, internal.MaxIncomingFriendRequests)
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, http.StatusBadRequest)
		return
	}

	cursorRelationship := backend.RequestedByUser
	cursorToken := ""
	if c := requestCursor(r); c != "" {
		cursorObj, err := cursor.Parse(c)
		if err != nil {
			gojiplus.ServePublicError(ctx, w, r, err, http.StatusBadRequest)
			return
		}
		cursorRelationship = cursorObj.Relationship
		cursorToken = cursorObj.Cursor
	}

	sort, err := requestSort(r, friendship.DescendingDir)
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, http.StatusBadRequest)
		return
	}

	resp, err := s.backend.GetIncomingRequestsV1(ctx, userID, cursorRelationship, cursorToken, sort, limit)
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, utils.ErrorStatus(err))
		return
	}

	encodedCursor := cursor.New(resp.CursorRelationship, resp.Cursor).Encode()
	gojiplus.ServeJSON(w, r, friendship.GetIncomingRequestsResponse{
		Requests: resp.Requests,
		Cursor:   encodedCursor,
		Total:    resp.Total,
	})
}

func (s *Server) getIncomingRequestsV0(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	required, err := gojiplus.FetchRequiredURLParams(ctx, "user_id")
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, http.StatusBadRequest)
		return
	}
	userID := required["user_id"]

	status := r.FormValue("status")
	if status == "" {
		status = "all"
	}

	limit, err := requestLimit(r, defaultIncomingRequestsLimit, internal.MaxIncomingFriendRequests)
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, http.StatusBadRequest)
		return
	}

	direction, err := requestSort(r, friendship.DescendingDir)
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, http.StatusBadRequest)
		return
	}

	friendsCursor := cursor.GetFriendRequestCursor{Relationship: backend.RequestedByUser, Cursor: ""}
	if c := requestCursor(r); c != "" {
		friendsCursor, err = cursor.Parse(c)
		if err != nil {
			gojiplus.ServePublicError(ctx, w, r, err, http.StatusBadRequest)
			return
		}
	} else {
		if status == "stranger" {
			friendsCursor.Relationship = backend.RequestedByStranger
		}
	}

	resp, err := s.backend.GetIncomingRequestsV0(
		ctx,
		userID,
		status,
		limit,
		friendsCursor.Relationship,
		friendsCursor.Cursor,
		direction,
	)
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, utils.ErrorStatus(err))
		return
	}

	encodedCursor := cursor.New(resp.CursorRelationship, resp.Cursor).Encode()
	gojiplus.ServeJSON(w, r, responses.FriendRequestListResponse{
		Requests:      resp.Associations,
		Strangers:     resp.Strangers,
		NonStrangers:  resp.NonStrangers,
		Cursor:        encodedCursor,
		Total:         resp.Total,
		IteratorToken: encodedCursor,
	})
}
