package api

import (
	"net/http"
	"strconv"

	"golang.org/x/net/context"

	"code.justin.tv/chat/friendship/app/api/responses"
	"code.justin.tv/chat/friendship/app/api/utils"
	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/golibs/logx"
)

const (
	// recSourceComputed is a value that indicates that the recommendations were computed by our Science team.
	//
	// Deprecated: All recommendations come from the same source now. This value should exist only as long as the API
	// endpoints that return it.
	recSourceComputed = "computed"
)

func (s *Server) getRecommendedV0(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	required, err := gojiplus.FetchRequiredURLParams(ctx, "user_id")
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, http.StatusBadRequest)
		return
	}
	userID := required["user_id"]

	recommended, err := s.backend.GetRecommended(ctx, userID)
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, utils.ErrorStatus(err))
		return
	}

	recommendedIDs := []int64{}
	for _, rec := range recommended {
		recID, err := strconv.ParseInt(rec.UserID, 10, 64)
		if err != nil {
			logx.Error(ctx, err)
			continue
		}
		recommendedIDs = append(recommendedIDs, recID)
	}

	gojiplus.ServeJSON(w, r, responses.RecommendedFriendsResponseV0{
		RecommendedIDs: recommendedIDs,
		Source:         recSourceComputed,
	})
}

func (s *Server) getRecommendedV1(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	required, err := gojiplus.FetchRequiredURLParams(ctx, "user_id")
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, http.StatusBadRequest)
		return
	}
	userID := required["user_id"]

	recommended, err := s.backend.GetRecommended(ctx, userID)
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, utils.ErrorStatus(err))
		return
	}

	recommendedFriends := []responses.RecommendedFriendV1{}
	for _, r := range recommended {
		recommendedFriends = append(recommendedFriends, responses.RecommendedFriendV1{
			UserID: r.UserID,
			Reason: r.Reason,
		})
	}

	gojiplus.ServeJSON(w, r, responses.RecommendedFriendsResponseV1{
		RecommendedFriends: recommendedFriends,
		Source:             recSourceComputed,
	})
}

func (s *Server) dismissRecommendedV0(ctx context.Context, w http.ResponseWriter, r *http.Request) {
	required, err := gojiplus.FetchRequiredURLParams(ctx, "user_id", "target_id")
	if err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, http.StatusBadRequest)
		return
	}
	userID, targetID := required["user_id"], required["target_id"]

	if err := s.backend.DismissRecommended(ctx, userID, targetID); err != nil {
		gojiplus.ServePublicError(ctx, w, r, err, utils.ErrorStatus(err))
		return
	}

	w.WriteHeader(http.StatusNoContent)
}
