package main

import (
	"errors"
	"flag"
	"fmt"
	_ "net/http/pprof"
	"time"

	"golang.org/x/net/context"

	"github.com/cactus/go-statsd-client/statsd"

	"code.justin.tv/chat/friendship/app/api"
	"code.justin.tv/chat/friendship/app/backend"
	"code.justin.tv/chat/friendship/config"
	"code.justin.tv/chat/golibs/clients/cohesion"
	"code.justin.tv/chat/golibs/gojiplus"
	"code.justin.tv/chat/golibs/logx"
	"code.justin.tv/chat/jsonconf"
	"code.justin.tv/web/cohesion/client/v2"
)

func main() {
	ctx := context.Background()

	confPath := flag.String("conf", "", "Path to config file")
	flag.Parse()
	if *confPath == "" {
		logx.Fatal(ctx, "empty config path")
	}
	var config config.Config
	if err := jsonconf.ReadFile(&config, *confPath); err != nil {
		logx.Fatal(ctx, err)
	}

	logx.InitDefaultLogger(logx.Config{
		RollbarToken: config.RollbarToken,
		RollbarEnv:   config.Environment,
	})
	defer logx.Wait()

	stats, err := statsd.NewBufferedClient(config.StatsHost, config.StatsPrefix, 1*time.Second, 512)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprintf("initializing statsd client: %v", err))
	}

	backender, err := backend.NewBackend(config, stats)
	if err != nil {
		logx.Fatal(ctx, err)
	}
	mux, err := api.NewServer(backender, stats)
	if err != nil {
		logx.Fatal(ctx, err)
	}

	go keepAlive(backender.Follows)

	gometricsStatsPrefix := fmt.Sprintf("%s.gometrics_%d", config.StatsPrefix, config.Port)
	gometricsStats, err := statsd.NewBufferedClient(config.StatsHost, gometricsStatsPrefix, 1*time.Second, 512)
	if err != nil {
		logx.Fatal(ctx, fmt.Sprintf("initializing statsd client: %v", err))
	}
	params := gojiplus.ServerParams{
		Port:           config.Port,
		DebugPort:      config.DebugPort,
		GometricsStats: gometricsStats,
		ReadTimeout:    3 * time.Second,
		WriteTimeout:   5 * time.Second,
	}
	if err := gojiplus.ListenAndServe(mux.Handler, params); err != nil {
		logx.Fatal(ctx, fmt.Sprintf("starting server: %v", err))
	}
}

// HACK:
// keepAlive hits cohesion-follows client at a 30s interval as a dummy keepalive.
func keepAlive(c cohesion.Client) {
	ctx := context.Background()
	ticker := time.NewTicker(30 * time.Second)
	for range ticker.C {
		_, err := c.Get(
			ctx,
			v2.Entity{ID: "45776283", Kind: backend.User},
			backend.Follows,
			v2.Entity{ID: "70617795", Kind: backend.User},
		)
		if err != nil {
		}
	}
}

func parseConfig(path string) (*config.Config, error) {
	if path == "" {
		return nil, errors.New("Empty config path")
	}
	config := &config.Config{}
	if err := jsonconf.ReadFile(config, path); err != nil {
		return nil, err
	}
	return config, nil
}
