package friendship

import (
	"time"

	"code.justin.tv/chat/friendship/app/api/responses"
	"code.justin.tv/foundation/twitchclient"
	"golang.org/x/net/context"
)

const (
	defaultStatSampleRate = 1.0

	// Requests to friendship should never take more than defaultCtxTimeout.
	// Callers can specify stricter deadlines if they'd like.
	defaultCtxTimeout = 3 * time.Second

	AscendingDir  = "ASC"
	DescendingDir = "DESC"
)

// Client exposes methods to make requests to the Friendship service.
type Client interface {
	// GetIncomingRequests returns open incoming friend requests for a user.
	GetIncomingRequests(ctx context.Context, userID string, params GetIncomingRequestsParams, reqOpts *twitchclient.ReqOpts) (GetIncomingRequestsResponse, error)
	// GetOutgoingRequests returns open outgoing friend requests for a user.
	GetOutgoingRequests(ctx context.Context, userID string, params GetOutgoingRequestsParams, reqOpts *twitchclient.ReqOpts) (GetOutgoingRequestsResponse, error)
	// Request sends a friend request to a user.
	Request(ctx context.Context, params RequestParams, reqOpts *twitchclient.ReqOpts) error
	// AcceptRequest accepts an incoming friend request for a user.
	AcceptRequest(ctx context.Context, userID, requesterID string, reqOpts *twitchclient.ReqOpts) error
	// RejectRequest declines an incoming friend request for a user.
	RejectRequest(ctx context.Context, userID, requesterID string, reqOpts *twitchclient.ReqOpts) error
	// BulkAddFriends forcefully friends a user with n other users. It does so regardless of the presence or acceptance of
	// friend requests, and will put users over the friendship cap if it needs to. It should only be used when importing
	// friendships with no intention of asking users to confirm each friend.
	BulkAddFriends(ctx context.Context, userID string, targetIDs []string, reqOpts *twitchclient.ReqOpts) error

	// GetNotificationCount returns the number of unread friend requests for a user.
	GetNotificationCount(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (uint, error)
	// ResetNotificationCount resets the unread notification counter for a user to 0.
	ResetNotificationCount(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) error

	// GetFriends returns a user's friends.
	GetFriends(ctx context.Context, userID string, params GetFriendsParams, reqOpts *twitchclient.ReqOpts) (*responses.FriendListResponse, error)
	// GetFriendship returns the current relationship between two users.
	GetFriendship(ctx context.Context, userID, targetID string, reqOpts *twitchclient.ReqOpts) (string, error)
	// IsFriend returns whether two users are friends.
	IsFriend(ctx context.Context, userID, targetID string, reqOpts *twitchclient.ReqOpts) (bool, error)
	// DeleteFriend removes a friend from a user's friends list.
	DeleteFriend(ctx context.Context, userID, targetID string, reqOpts *twitchclient.ReqOpts) error

	// GetRecommendedFriends returns a user's recommended friends.
	GetRecommendedFriends(ctx context.Context, userID string, reqOpts *twitchclient.ReqOpts) (*responses.RecommendedFriendsResponseV1, error)
	// DismissRecommendation dismisses a friend recommendation.
	DismissRecommendation(ctx context.Context, userID, targetID string, reqOpts *twitchclient.ReqOpts) error

	// IsFamiliar returns whether a user is familiar to another user from the given criteria.
	// An empty criteria slice defaults to checking all criteria.
	IsFamiliar(ctx context.Context, userID, targetID string, criteria []string, reqOpts *twitchclient.ReqOpts) (string, bool, error)
}

// NewClient creates a client for the friendship service.
func NewClient(conf twitchclient.ClientConf) (Client, error) {
	if conf.TimingXactName == "" {
		conf.TimingXactName = "friendship"
	}
	c, err := twitchclient.NewClient(conf)
	if err != nil {
		return nil, err
	}
	return &clientImpl{Client: c}, nil
}

type clientImpl struct {
	twitchclient.Client
}
