package friendship

import (
	"fmt"
	"net/url"
	"strconv"
	"time"

	"code.justin.tv/foundation/twitchclient"
	"golang.org/x/net/context"
)

type GetOutgoingRequestsResponse struct {
	Requests []OutgoingRequest `json:"requests"`
	Cursor   string            `json:"cursor"`
	Total    uint              `json:"total"`
}

type OutgoingRequest struct {
	UserID            string    `json:"user_id"`
	RequestedAt       time.Time `json:"requested_at"`
	IsRecommended     bool      `json:"is_recommended"`
	IsStranger        bool      `json:"is_stranger"`
	NonStrangerReason string    `json:"non_stranger_reason"`
}

type GetOutgoingRequestsParams struct {
	Limit  *int
	Cursor *string
	Sort   *string
}

func (c *clientImpl) GetOutgoingRequests(
	ctx context.Context, userID string, params GetOutgoingRequestsParams, reqOpts *twitchclient.ReqOpts,
) (GetOutgoingRequestsResponse, error) {
	q := url.Values{}
	if params.Limit != nil {
		q.Add("limit", strconv.Itoa(*params.Limit))
	}
	if params.Cursor != nil {
		q.Add("cursor", *params.Cursor)
	}
	if params.Sort != nil {
		q.Add("sort", *params.Sort)
	}

	u := &url.URL{
		Path:     fmt.Sprintf("/v1/users/%s/friends/outgoing_requests", userID),
		RawQuery: q.Encode(),
	}
	req, err := c.NewRequest("GET", u.String(), nil)
	if err != nil {
		return GetOutgoingRequestsResponse{}, err
	}

	combinedReqOpts := twitchclient.MergeReqOpts(reqOpts, twitchclient.ReqOpts{
		StatName:       "service.friendship.get_outgoing_requests",
		StatSampleRate: defaultStatSampleRate,
	})
	ctx, cancel := context.WithTimeout(ctx, defaultCtxTimeout)
	defer cancel()

	var decoded GetOutgoingRequestsResponse
	if _, err := c.DoJSON(ctx, &decoded, req, combinedReqOpts); err != nil {
		return GetOutgoingRequestsResponse{}, err
	}
	return decoded, nil
}
