package duplo

import (
	"fmt"
	"net/http"
	"net/url"
	"time"

	"strings"

	"strconv"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/feeds-common/entity"
	"github.com/google/go-querystring/query"
	"golang.org/x/net/context"
)

// Post is a basic post
type Post struct {
	ID        string     `json:"id"`
	UserID    string     `json:"user_id"`
	Body      string     `json:"body"`
	CreatedAt time.Time  `json:"created_at"`
	DeletedAt *time.Time `json:"deleted_at,omitempty"`
	AudreyID  string     `json:"audrey_id,omitempty"`
	Emotes    *[]Emote   `json:"emotes,omitempty"`
}

// SharesSummary contains summary information about an object that may be shared
type SharesSummary struct {
	ParentEntity entity.Entity `json:"parent_entity"`
	Total        int           `json:"total"`
}

// SharesSummaries is a wrapper around a list of ShareSummary results
type SharesSummaries struct {
	Items []*SharesSummary `json:"items"`
}

// Posts is a wrapper around a list of post results
type Posts struct {
	Items []*Post `json:"items"`
}

// Shares is a wrapper around a list of share results
type Shares struct {
	Items []*Share `json:"items"`
}

// Comment is a basic comment
type Comment struct {
	ID            string        `json:"id"`
	ParentEntity  entity.Entity `json:"parent_entity"`
	UserID        string        `json:"user_id"`
	Body          string        `json:"body"`
	CreatedAt     time.Time     `json:"created_at"`
	DeletedAt     *time.Time    `json:"deleted_at,omitempty"`
	AudreyID      string        `json:"audrey_id,omitempty"`
	NeedsApproval bool          `json:"needs_approval"`
}

// Comments is the comment object that is returned for paginated comments
type Comments struct {
	Items  []*Comment `json:"items"`
	Cursor string     `json:"cursor"`
}

// PaginatedPostIDs is the list of post IDs that can be cursor'd
type PaginatedPostIDs struct {
	PostIDs []string `json:"post_ids"`
	Cursor  string   `json:"cursor"`
}

// CommentsSummary is a summary of comments for a parent entity
type CommentsSummary struct {
	ParentEntity entity.Entity `json:"parent_entity"`
	Total        int           `json:"total"`
}

// CommentsSummaries is the response wrapper for a list of CommentsSummaries
type CommentsSummaries struct {
	Items []*CommentsSummary `json:"items"`
}

// Reactions is a basic reaction
type Reactions struct {
	ParentEntity entity.Entity `json:"parent_entity"`
	UserID       string        `json:"user_id"`
	EmoteIDs     []string      `json:"emote_ids"`
}

// ReactionsSummary is a basic reaction summary
type ReactionsSummary struct {
	ParentEntity   entity.Entity            `json:"parent_entity"`
	EmoteSummaries map[string]*EmoteSummary `json:"emote_summaries,omitempty"`

	// deprecated
	Emotes map[string]int `json:"emotes"`
}

// ReactionsSummaries is a wrapper around a list of reaction summaries
type ReactionsSummaries struct {
	Items []*ReactionsSummary `json:"items"`
}

// EmoteSummary contains a true count of reactions of an emote type and a subset of the userids that have actually
// reacted
type EmoteSummary struct {
	// The real total number of reactions of this emote type
	Count int `json:"count"`
	// Some of the user IDs that reacted to this emote
	UserIDs []string `json:"user_ids,omitempty"`
}

// Config configures duplo http client
type Config struct {
	Host func() string
}

// Load configuration information
func (c *Config) Load(dconf *distconf.Distconf) error {
	c.Host = dconf.Str("duplo.http_endpoint", "").Get
	return nil
}

// Client is used by third parties to connect and interact with Audrey
type Client struct {
	RequestDoer    clients.RequestDoer
	NewHTTPRequest clients.NewHTTPRequest
	Config         *Config
}

type errResponse struct {
	StatusCode int    `json:"-"`
	Message    string `json:"message"`
}

func (e errResponse) Error() string {
	return fmt.Sprintf("%d: %s", e.StatusCode, e.Message)
}

func (c *Client) http(ctx context.Context, method string, path string, queryParams url.Values, body interface{}, into interface{}) error {
	return clients.DoHTTP(ctx, c.RequestDoer, method, c.Config.Host()+path, queryParams, body, into, c.NewHTTPRequest)
}

// PostID contains the GUID for the post
type PostID struct {
	ID string `json:"id"`
}

// GetPostIDByLegacyID returns the ID for the given legacy ID string
func (c *Client) GetPostIDByLegacyID(ctx context.Context, legacyID string) (*PostID, error) {
	path := "/v1/posts/legacy/" + legacyID + "/id"
	var ret PostID
	if err := c.http(ctx, "GET", path, nil, nil, &ret); err != nil {
		return nil, err
	}
	return &ret, nil
}

// GetPosts returns multiple posts for a given list of post ids, or nil
func (c *Client) GetPosts(ctx context.Context, postIDs []string) (*Posts, error) {
	return c.GetPostsWithOptions(ctx, postIDs, nil)
}

// GetPostsOptions are optional configuration for getting multiple posts at once
type GetPostsOptions struct {
	Shares     bool
	ShareUsers []string
}

// GetPostsWithOptions returns multiple posts for a given list of post ids, or nil.  Allows also populating the
// posts with share information.
func (c *Client) GetPostsWithOptions(ctx context.Context, postIDs []string, options *GetPostsOptions) (*Posts, error) {
	values := make(url.Values)
	values.Add("ids", strings.Join(postIDs, ","))
	if options != nil {
		if options.Shares {
			values.Add("shares", "true")
		}
		if len(options.ShareUsers) > 0 {
			values.Add("share_users", strings.Join(options.ShareUsers, ","))
		}
	}
	var ret Posts
	if err := c.http(ctx, "GET", "/v1/posts", values, nil, &ret); err != nil {
		return nil, err
	}
	return &ret, nil
}

// Emote is the emote object for the API
type Emote struct {
	ID    int `json:"id"`
	Start int `json:"start"`
	End   int `json:"end"`
	Set   int `json:"set"`
}

// CreateShareOptions is a struct for future optional parameters to CreateShare
type CreateShareOptions struct {
}

// Share is the returned object for a database share
type Share struct {
	ID           string        `json:"id"`
	UserID       string        `json:"user_id"`
	TargetEntity entity.Entity `json:"target_entity"`
	CreatedAt    time.Time     `json:"created_at"`
	DeletedAt    *time.Time    `json:"deleted_at,omitempty"`
}

// ShareSummary contains total share amount for a single target entity
type ShareSummary struct {
	ParentEntity entity.Entity `json:"parent_entity"`
	Total        int           `json:"total"`
}

// CreateShare creates a share for an entity.  options are currently ignored
func (c *Client) CreateShare(ctx context.Context, userID string, targetEntity entity.Entity, options *CreateShareOptions) (*Share, error) {
	path := "/v1/shares"
	httpBody := struct {
		UserID       string        `json:"user_id"`
		TargetEntity entity.Entity `json:"target_entity"`
	}{
		UserID:       userID,
		TargetEntity: targetEntity,
	}

	var ret Share
	if err := c.http(ctx, "POST", path, nil, httpBody, &ret); err != nil {
		return nil, err
	}
	return &ret, nil
}

// GetSharesByAuthor returns all shares someone has authored on a list of target entities
func (c *Client) GetSharesByAuthor(ctx context.Context, authorID string, targetEntities []entity.Entity) (*Shares, error) {
	path := "/v1/shares_by_author"
	values := make(url.Values)
	values.Add("author_id", authorID)
	values.Add("target_entities", strings.Join(entity.EncodeAll(targetEntities), ","))

	var ret Shares
	if err := c.http(ctx, "GET", path, values, nil, &ret); err != nil {
		return nil, err
	}
	return &ret, nil
}

// GetShare returns a share.  Will return nil, nil if the share doesn't exist
func (c *Client) GetShare(ctx context.Context, shareID string) (*Share, error) {
	path := "/v1/shares/" + shareID

	var ret Share
	if err := c.http(ctx, "GET", path, nil, nil, &ret); err != nil {
		if clients.ErrorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}
	return &ret, nil
}

// GetSharesSummaries returns summary information about the total number of shares a list of items have
func (c *Client) GetSharesSummaries(ctx context.Context, parentEntities []entity.Entity) (*SharesSummaries, error) {
	path := "/v1/shares_summaries"
	values := make(url.Values)
	values.Add("parent_entities", strings.Join(entity.EncodeAll(parentEntities), ","))

	var ret SharesSummaries
	if err := c.http(ctx, "GET", path, values, nil, &ret); err != nil {
		return nil, err
	}
	return &ret, nil
}

// GetShares returns multiple share for a given list of share ids, or nil
func (c *Client) GetShares(ctx context.Context, shareIDs []string) (*Shares, error) {
	path := "/v1/shares?ids=" + strings.Join(shareIDs, ",")
	var ret Shares
	if err := c.http(ctx, "GET", path, nil, nil, &ret); err != nil {
		return nil, err
	}
	return &ret, nil
}

// DeleteShare removes a share.  Will return nil, nil if the share doesn't exist
func (c *Client) DeleteShare(ctx context.Context, shareID string) (*Share, error) {
	path := "/v1/shares/" + shareID

	var ret Share
	if err := c.http(ctx, "DELETE", path, nil, nil, &ret); err != nil {
		if clients.ErrorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}
	return &ret, nil
}

// CreatePostOptions specifies the optional parameters of a CreatePost operation
type CreatePostOptions struct {
	// ID of the post from the Audrey DB
	AudreyID string
	// The time that the post was creatd.  Used for migration and will be removed post migration
	CreateAt *time.Time
	// The time that the post was deleted.  Used for migration and will be removed post migration
	DeletedAt *time.Time
	// Emotes are an optional list of emotes that exist on the post.  missing/nil means they aren't calculated
	// while empty means they are calculated and there aren't any
	Emotes *[]Emote
}

type createPostBody struct {
	UserID    string     `json:"user_id"`
	Body      string     `json:"body"`
	AudreyID  string     `json:"audrey_id,omitempty"`
	CreateAt  *time.Time `json:"created_at,omitempty"`
	DeletedAt *time.Time `json:"deleted_at,omitempty"`
	Emotes    *[]Emote   `json:"emotes,omitempty"`
}

// CreatePost creates a post with the given content
func (c *Client) CreatePost(ctx context.Context, userID string, body string, options *CreatePostOptions) (*Post, error) {
	path := "/v1/posts"
	httpBody := createPostBody{
		UserID: userID,
		Body:   body,
	}
	if options != nil {
		httpBody.AudreyID = options.AudreyID
		httpBody.CreateAt = options.CreateAt
		httpBody.DeletedAt = options.DeletedAt
		httpBody.Emotes = options.Emotes
	}

	var ret Post
	if err := c.http(ctx, "POST", path, nil, httpBody, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// UpdatePostOptions specifies the optional parameters of a UpdatePost operation
type UpdatePostOptions struct {
	// Emotes are an optional list of emotes that exist on the post.  missing/nil means they aren't calculated
	// while empty means they are calculated and there aren't any
	Emotes *[]Emote
}

type updatePostBody struct {
	Emotes *[]Emote `json:"emotes,omitempty"`
}

// UpdatePost updates a post
func (c *Client) UpdatePost(ctx context.Context, postID string, options UpdatePostOptions) error {
	path := "/v1/posts/" + postID
	httpBody := updatePostBody{
		Emotes: options.Emotes,
	}

	return c.http(ctx, "PUT", path, nil, httpBody, nil)
}

// GetPost returns the post for the given ID, or nil
func (c *Client) GetPost(ctx context.Context, postID string) (*Post, error) {
	path := "/v1/posts/" + postID
	var ret Post
	if err := c.http(ctx, "GET", path, nil, nil, &ret); err != nil {
		if clients.ErrorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}
	return &ret, nil
}

// DeletePostOptions specifies the optional parameters of a DeletePost operation
type DeletePostOptions struct {
	// The time that the post was deleted.  Used for migration and will be removed post migration
	DeletedAt *time.Time
}

// DeletePost deletes the post for the given ID
func (c *Client) DeletePost(ctx context.Context, postID string, options *DeletePostOptions) (*Post, error) {
	path := "/v1/posts/" + postID
	query := url.Values{}
	if options != nil {
		if options.DeletedAt != nil {
			query.Add("deleted_at", options.DeletedAt.Format(time.RFC3339))
		}
	}

	var ret Post
	if err := c.http(ctx, "DELETE", path, query, nil, &ret); err != nil {
		if clients.ErrorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}
	return &ret, nil
}

// CommentID contains the GUID for the post
type CommentID struct {
	ID string `json:"id"`
}

// GetCommentIDByLegacyID returns the ID for the given legacy ID string
func (c *Client) GetCommentIDByLegacyID(ctx context.Context, legacyID string) (*CommentID, error) {
	path := "/v1/comments/legacy/" + legacyID + "/id"
	var ret CommentID
	if err := c.http(ctx, "GET", path, nil, nil, &ret); err != nil {
		return nil, err
	}
	return &ret, nil
}

// CreateCommentOptions specifies the optional parameters of a CreateComment operation
type CreateCommentOptions struct {
	// ID of the post from the Audrey DB.
	AudreyID string `json:"audrey_id,omitempty"`
	// The time that the post was created.  Used for migration and will be removed post migration
	CreateAt *time.Time `json:"created_at,omitempty"`
	// The time that the post was deleted.  Used for migration and will be removed post migration
	DeletedAt *time.Time `json:"deleted_at,omitempty"`
	// Whether or not the comment needs approval to be 'posted'.
	NeedsApproval bool `json:"needs_approval"`
}

type createCommentBody struct {
	ParentEntity  entity.Entity `json:"parent_entity"`
	UserID        string        `json:"user_id"`
	Body          string        `json:"body"`
	AudreyID      string        `json:"audrey_id,omitempty"`
	CreateAt      *time.Time    `json:"created_at,omitempty"`
	DeletedAt     *time.Time    `json:"deleted_at,omitempty"`
	NeedsApproval bool          `json:"needs_approval"`
}

// CreateComment create a comment with the given context for the specified parent entity ID
func (c *Client) CreateComment(ctx context.Context, parentEntity entity.Entity, userID string, body string, options *CreateCommentOptions) (*Comment, error) {
	path := "/v1/comments"
	httpBody := createCommentBody{
		ParentEntity: parentEntity,
		UserID:       userID,
		Body:         body,
	}
	if options != nil {
		httpBody.AudreyID = options.AudreyID
		httpBody.CreateAt = options.CreateAt
		httpBody.DeletedAt = options.DeletedAt
		httpBody.NeedsApproval = options.NeedsApproval
	}

	var ret Comment
	if err := c.http(ctx, "POST", path, nil, httpBody, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// GetComment gets the comment for the given ID
func (c *Client) GetComment(ctx context.Context, commentID string) (*Comment, error) {
	path := "/v1/comments/" + commentID
	var ret Comment
	if err := c.http(ctx, "GET", path, nil, nil, &ret); err != nil {
		if clients.ErrorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}

	return &ret, nil
}

// GetCommentsSummariesByParent gets the comments summaries for the given parent entities
func (c *Client) GetCommentsSummariesByParent(ctx context.Context, parentEntities []entity.Entity) (*CommentsSummaries, error) {
	path := "/v1/comments/summary"
	params := url.Values{}
	params.Add("parent_entity", strings.Join(entity.EncodeAll(parentEntities), ","))

	var ret CommentsSummaries
	if err := c.http(ctx, "GET", path, params, nil, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// GetCommentsByParentOptions specifies the optional parameters of a GetComment operation
type GetCommentsByParentOptions struct {
	UserID string `url:"user_id,omitempty"`
	Cursor string `url:"cursor,omitempty"`
	Limit  int    `url:"limit,omitempty"`
}

// GetCommentsByParent gets all comments associated with the given parent entity
func (c *Client) GetCommentsByParent(ctx context.Context, parentEntity entity.Entity, options *GetCommentsByParentOptions) (*Comments, error) {
	path := "/v1/comments"

	query, err := query.Values(options)
	if err != nil {
		return nil, err
	}
	query.Add("parent_entity", parentEntity.Encode())

	var ret Comments
	if err := c.http(ctx, "GET", path, query, nil, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// UpdateCommentParams specifies the optional parameters of a EditComment operation
type UpdateCommentParams struct {
	NeedsApproval *bool `json:"needs_approval,omitempty"`
}

// UpdateComment updates a comment
func (c *Client) UpdateComment(ctx context.Context, commentID string, params *UpdateCommentParams) (*Comment, error) {
	path := "/v1/comments/" + commentID

	var ret Comment
	if err := c.http(ctx, "PUT", path, nil, params, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// DeleteCommentOptions specifies the optional parameters of a DeleteComment operation
type DeleteCommentOptions struct {
	// The time that the comment was deleted.  Used for migration and will be removed post migration
	DeletedAt *time.Time
}

// GetPostIDsByUserOptions controls optional query params for GetPostIDsByUser
type GetPostIDsByUserOptions struct {
	// Limit how many to return
	Limit int
	// Cursor to continue a previous request from
	Cursor string
}

// DeleteComment deletes the comment for the given ID
func (c *Client) DeleteComment(ctx context.Context, commentID string, options *DeleteCommentOptions) (*Comment, error) {
	path := "/v1/comments/" + commentID
	query := url.Values{}
	if options != nil {
		if options.DeletedAt != nil {
			query.Add("deleted_at", options.DeletedAt.Format(time.RFC3339))
		}
	}

	var ret Comment
	if err := c.http(ctx, "DELETE", path, query, nil, &ret); err != nil {
		if clients.ErrorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}

	return &ret, nil
}

// DeleteCommentsByParentAndUser deletes all comments for a given parent entity and user ID
func (c *Client) DeleteCommentsByParentAndUser(ctx context.Context, parentEntity entity.Entity, userID string) error {
	path := fmt.Sprintf("/v1/comments/by_parent_and_user/%s/%s", parentEntity, userID)
	return c.http(ctx, "DELETE", path, nil, nil, nil)
}

// GetReactions gets the user reactions for a given Parent Entity ID
func (c *Client) GetReactions(ctx context.Context, parentEntity entity.Entity, userID string) (*Reactions, error) {
	path := fmt.Sprintf("/v1/reactions/%s/%s", parentEntity.Encode(), userID)
	var ret Reactions
	if err := c.http(ctx, "GET", path, nil, nil, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// GetPostIDsByUser gets posts from a user
func (c *Client) GetPostIDsByUser(ctx context.Context, userID string, options *GetPostIDsByUserOptions) (*PaginatedPostIDs, error) {
	path := fmt.Sprintf("/v1/posts/ids_by_user/%s", userID)
	query := url.Values{}
	if options != nil {
		if options.Cursor != "" {
			query.Set("cursor", options.Cursor)
		}
		if options.Limit != 0 {
			query.Set("limit", strconv.Itoa(options.Limit))
		}
	}
	var ret PaginatedPostIDs
	if err := c.http(ctx, "GET", path, query, nil, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// CreateReaction adds a reaction for a given user ID to an existing parent entity ID
func (c *Client) CreateReaction(ctx context.Context, parentEntity entity.Entity, userID string, emoteID string) (*Reactions, error) {
	path := fmt.Sprintf("/v1/reactions/%s/%s/%s", parentEntity.Encode(), userID, emoteID)

	var ret Reactions
	if err := c.http(ctx, "PUT", path, nil, nil, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// DeleteReaction remotes a reaction for a given user ID to an existing parent entity ID
func (c *Client) DeleteReaction(ctx context.Context, parentEntity entity.Entity, userID string, emoteID string) (*Reactions, error) {
	path := fmt.Sprintf("/v1/reactions/%s/%s/%s", parentEntity.Encode(), userID, emoteID)

	var ret Reactions
	if err := c.http(ctx, "DELETE", path, nil, nil, &ret); err != nil {
		if clients.ErrorCode(err) == http.StatusNotFound {
			return nil, nil
		}
		return nil, err
	}

	return &ret, nil
}

// GetReactionsSummary gets the summary reactions for a given Parent Entity ID
func (c *Client) GetReactionsSummary(ctx context.Context, parentEntity entity.Entity) (*ReactionsSummary, error) {
	path := fmt.Sprintf("/v1/reactions/summary/%s", parentEntity.Encode())
	var ret ReactionsSummary
	if err := c.http(ctx, "GET", path, nil, nil, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}

// GetReactionsSummariesByParentOptions controls optional query params for GetReactionsSummariesByParent
type GetReactionsSummariesByParentOptions struct {
	// The users that you want use to check whether they've reacted or not
	UserIDs []string
}

// GetReactionsSummariesByParent gets the summaries of the reactions for the parent entities
func (c *Client) GetReactionsSummariesByParent(ctx context.Context, parentEntities []entity.Entity, options *GetReactionsSummariesByParentOptions) (*ReactionsSummaries, error) {
	path := "/v1/reactions/summary"
	params := url.Values{}
	params.Add("parent_entity", strings.Join(entity.EncodeAll(parentEntities), ","))
	if options != nil {
		if len(options.UserIDs) > 0 {
			params.Add("user_ids", strings.Join(options.UserIDs, ","))
		}
	}

	var ret ReactionsSummaries
	if err := c.http(ctx, "GET", path, params, nil, &ret); err != nil {
		return nil, err
	}

	return &ret, nil
}
