// +build integration

package duplo_test

import (
	"testing"
	"time"

	"math/rand"
	"strconv"

	"sync"

	"code.justin.tv/feeds/clients/duplo"
	"code.justin.tv/feeds/errors"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/service-common"
	. "github.com/smartystreets/goconvey/convey"
	"golang.org/x/net/context"
)

func newSetup() *TestSetup {
	return &TestSetup{
		Common: service_common.ConfigCommon{
			Team:          "feeds",
			Service:       "client_int_tests",
			BaseDirectory: "../",
		},
	}
}

var r *rand.Rand
var mu sync.Mutex

func init() {
	r = rand.New(rand.NewSource(time.Now().UnixNano()))
}

func newTestUserID() string {
	mu.Lock()
	defer mu.Unlock()
	return strconv.FormatInt(int64(r.Int31()), 10)
}

func timestamp() string {
	return strconv.FormatInt(time.Now().UnixNano(), 10)
}

func TestClientIntegration_Post(t *testing.T) {
	t.Parallel()
	Convey("With integration client", t, func() {
		setup := newSetup()
		So(setup.Setup(), ShouldBeNil)
		Reset(func() {
			So(setup.Close(), ShouldBeNil)
		})
		ctx := setup.Context
		c := setup.Client
		t.Log("using client addr ", c.Config.Host())
		testUserID := newTestUserID()

		Convey("Update a post that doesn't exist should fail", func() {
			err := c.UpdatePost(ctx, strconv.Itoa(int(time.Now().UnixNano())), duplo.UpdatePostOptions{Emotes: &[]duplo.Emote{
				{
					ID:    1,
					Start: 2,
					End:   3,
					Set:   4,
				},
			}})
			So(err, ShouldNotBeNil)
		})

		Convey("Should be able to create and get a post", func() {
			testBody := "TestClientIntegration_Post " + timestamp()
			created, err := c.CreatePost(ctx, testUserID, testBody, nil)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.ID, ShouldNotBeNil)
			So(created.Body, ShouldEqual, testBody)

			post, err := c.GetPost(ctx, created.ID)
			So(err, ShouldBeNil)
			So(post, ShouldNotBeNil)
			So(post.Body, ShouldEqual, testBody)
			So(post.Emotes, ShouldBeNil)

			Convey("post should update with an emote", func() {
				err := c.UpdatePost(ctx, post.ID, duplo.UpdatePostOptions{Emotes: &[]duplo.Emote{
					{
						ID:    1,
						Start: 2,
						End:   3,
						Set:   4,
					},
				}})
				So(err, ShouldBeNil)
				post, err := c.GetPost(ctx, created.ID)
				So(err, ShouldBeNil)
				So(post, ShouldNotBeNil)
				So(post.Body, ShouldEqual, testBody)
				So(post.Emotes, ShouldNotBeNil)
				So(len(*post.Emotes), ShouldEqual, 1)
				So((*post.Emotes)[0].Start, ShouldEqual, 2)
			})

			Convey("post should update with zero emotes", func() {
				err := c.UpdatePost(ctx, post.ID, duplo.UpdatePostOptions{Emotes: &[]duplo.Emote{}})
				So(err, ShouldBeNil)
				post, err := c.GetPost(ctx, created.ID)
				So(err, ShouldBeNil)
				So(post, ShouldNotBeNil)
				So(post.Body, ShouldEqual, testBody)
				So(post.Emotes, ShouldNotBeNil)
				So(len(*post.Emotes), ShouldEqual, 0)
			})

			Convey("post should remove", func() {
				deleted, err := c.DeletePost(ctx, created.ID, nil)
				So(err, ShouldBeNil)
				So(deleted, ShouldNotBeNil)
				So(deleted.DeletedAt, ShouldNotBeNil)

				post, err = c.GetPost(ctx, created.ID)
				So(err, ShouldBeNil)
				So(post, ShouldBeNil)

				posts, err := c.GetPosts(ctx, []string{created.ID})
				So(err, ShouldBeNil)
				So(len(posts.Items), ShouldEqual, 0)
			})

			Convey("Post should be fetchable by userID", func() {
				ids, err := tryXTimes(3, func() (interface{}, error) {
					a, b := c.GetPostIDsByUser(ctx, testUserID, &duplo.GetPostIDsByUserOptions{Limit: 2})
					if b != nil || len(a.PostIDs) == 0 {
						return nil, errors.New("Unable to fetch so far")
					}
					return a, b
				})
				So(err, ShouldBeNil)
				So(len(ids.(*duplo.PaginatedPostIDs).PostIDs), ShouldEqual, 1)
				So(ids.(*duplo.PaginatedPostIDs).PostIDs[0], ShouldEqual, created.ID)
			})
		})

		Convey("Should be create a post with an audrey_id and get the ID back", func() {
			testBody := "TestClientIntegration_Post_Audrey " + timestamp()
			testAudreyID := "test:audrey:" + timestamp()
			created, err := c.CreatePost(ctx, testUserID, testBody, &duplo.CreatePostOptions{
				AudreyID: testAudreyID,
			})
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.ID, ShouldNotBeNil)

			postInt, err := tryXTimes(3, func() (interface{}, error) {
				return c.GetPostIDByLegacyID(ctx, testAudreyID)
			})
			So(err, ShouldBeNil)
			postID := postInt.(*duplo.PostID)
			So(postID, ShouldNotBeNil)
			So(postID.ID, ShouldEqual, created.ID)
		})

		Convey("Should be able to get multiple posts", func() {
			testBody1 := "TestClientIntegration_Post " + timestamp()
			c1, err := c.CreatePost(ctx, testUserID, testBody1, nil)
			So(err, ShouldBeNil)
			So(c1, ShouldNotBeNil)
			So(c1.ID, ShouldNotBeNil)
			So(c1.Body, ShouldEqual, testBody1)

			testBody2 := "TestClientIntegration_Post " + timestamp()
			c2, err := c.CreatePost(ctx, testUserID, testBody2, nil)
			So(err, ShouldBeNil)
			So(c2, ShouldNotBeNil)
			So(c2.ID, ShouldNotBeNil)
			So(c2.Body, ShouldEqual, testBody2)

			posts, err := c.GetPosts(ctx, []string{c1.ID, c2.ID})
			So(err, ShouldBeNil)
			So(posts, ShouldNotBeNil)
			So(len(posts.Items), ShouldEqual, 2)
			So(c1, ShouldBeIn, posts.Items)
			So(c2, ShouldBeIn, posts.Items)
			Convey("and if one is shared it should be discovered w/ GetPosts", func() {
				c1Ent := entity.New(entity.NamespacePost, c1.ID)
				c1Share, err := c.CreateShare(ctx, testUserID, c1Ent, nil)
				So(err, ShouldBeNil)
				So(c1Share, ShouldNotBeNil)

				posts2, err := c.GetPostsWithOptions(ctx, []string{c1.ID, c2.ID}, &duplo.GetPostsOptions{
					Shares:     true,
					ShareUsers: []string{testUserID, testUserID + "1"},
				})
				So(err, ShouldBeNil)
				So(posts2, ShouldNotBeNil)
			})
		})
	})
}

func tryXTimes(count int, f func() (interface{}, error)) (interface{}, error) {
	var a interface{}
	var b error
	for i := 0; i < count; i++ {
		a, b = f()
		if b == nil {
			return a, b
		}
	}
	return a, b
}

func TestClientIntegration_Comment(t *testing.T) {
	t.Parallel()
	Convey("With integration client", t, func() {
		setup := newSetup()
		So(setup.Setup(), ShouldBeNil)
		Reset(func() {
			So(setup.Close(), ShouldBeNil)
		})
		ctx := setup.Context
		c := setup.Client
		testUserID := newTestUserID()

		Convey("Should be able to create, get, and delete a comment", func() {
			parent := entity.New("test_parent", timestamp())
			testBody := "TestClientIntegration_Comment " + timestamp()
			created, err := c.CreateComment(ctx, parent, testUserID, testBody, nil)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.ID, ShouldNotBeNil)
			So(created.ParentEntity, ShouldResemble, parent)
			So(created.Body, ShouldEqual, testBody)

			commentI, err := tryXTimes(3, func() (interface{}, error) {
				return c.GetComment(ctx, created.ID)
			})
			So(err, ShouldBeNil)
			comment := commentI.(*duplo.Comment)
			So(comment, ShouldNotBeNil)
			So(comment.Body, ShouldEqual, testBody)

			deleted, err := c.DeleteComment(ctx, created.ID, nil)
			So(err, ShouldBeNil)
			So(deleted, ShouldNotBeNil)
			So(deleted.DeletedAt, ShouldNotBeNil)

			comment, err = c.GetComment(ctx, created.ID)
			So(err, ShouldBeNil)
			So(comment, ShouldBeNil)
		})

		Convey("Should be able to edit a comment's needs approval", func() {
			parent := entity.New("test_parent", timestamp())
			testBody := "TestClientIntegration_Comment " + timestamp()
			createOptions := &duplo.CreateCommentOptions{
				NeedsApproval: true,
			}

			created, err := c.CreateComment(ctx, parent, testUserID, testBody, createOptions)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.NeedsApproval, ShouldBeTrue)

			needsApproval := false
			editOptions := &duplo.UpdateCommentParams{
				NeedsApproval: &needsApproval,
			}
			edited, err := c.UpdateComment(ctx, created.ID, editOptions)
			So(err, ShouldBeNil)
			So(edited, ShouldNotBeNil)
			So(edited.ID, ShouldEqual, created.ID)
			So(edited.NeedsApproval, ShouldBeFalse)
		})

		Convey("Should be able to delete comments by parent and user ID", func() {
			parent := entity.New("test_parent", timestamp())
			testBody := "TestClientIntegration_Comment " + timestamp()
			created, err := c.CreateComment(ctx, parent, testUserID, testBody, nil)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)

			err = c.DeleteCommentsByParentAndUser(ctx, parent, testUserID)
			So(err, ShouldBeNil)

			comment, err := c.GetComment(ctx, created.ID)
			So(err, ShouldBeNil)
			So(comment, ShouldBeNil)
		})

		Convey("Should be able to create a comment with an audrey_id and get the ID back", func() {
			parent := entity.New("test_parent", timestamp())
			testBody := "TestClientIntegration_Comment_Audrey " + timestamp()
			testAudreyID := "test:audrey:" + timestamp()
			created, err := c.CreateComment(ctx, parent, testUserID, testBody, &duplo.CreateCommentOptions{
				AudreyID: testAudreyID,
			})
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.ID, ShouldNotBeNil)

			commentID, err := tryXTimes(3, func() (interface{}, error) {
				return c.GetCommentIDByLegacyID(ctx, testAudreyID)
			})
			So(err, ShouldBeNil)
			So(commentID, ShouldNotBeNil)
			So(commentID.(*duplo.CommentID).ID, ShouldEqual, created.ID)
		})

		Convey("Should be able to get comments and comments summaries by parent entity", func() {
			parent := entity.New("test_parent", timestamp())
			testBody := "TestClientIntegration_Comment " + timestamp()
			created, err := c.CreateComment(ctx, parent, testUserID, testBody, nil)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.ID, ShouldNotBeNil)
			So(created.ParentEntity, ShouldResemble, parent)
			So(created.Body, ShouldEqual, testBody)

			comments, err := c.GetCommentsByParent(ctx, parent, nil)
			So(err, ShouldBeNil)
			So(comments.Items, ShouldNotBeNil)
			So(comments.Items, ShouldContain, created)

			summaries, err := c.GetCommentsSummariesByParent(ctx, []entity.Entity{parent})
			So(err, ShouldBeNil)
			So(summaries.Items, ShouldNotBeNil)
			So(len(summaries.Items), ShouldEqual, 1)
			So(summaries.Items[0].ParentEntity, ShouldResemble, parent)
			So(summaries.Items[0].Total, ShouldEqual, 1)
		})
	})
}

func TestClientIntegration_Share(t *testing.T) {
	t.Parallel()
	Convey("With integration client", t, func() {
		setup := newSetup()
		So(setup.Setup(), ShouldBeNil)
		Reset(func() {
			So(setup.Close(), ShouldBeNil)
		})
		ctx := setup.Context
		c := setup.Client
		testUserID := newTestUserID()
		testEntityID := entity.New("test_post", newTestUserID())

		Convey("Should be able to create a share", func() {
			share, err := c.CreateShare(ctx, testUserID, testEntityID, nil)
			So(err, ShouldBeNil)
			So(share.ID, ShouldNotEqual, "")
			Convey("and fetch it back", func() {
				share2, err := c.GetShare(ctx, share.ID)
				So(err, ShouldBeNil)
				So(share2.TargetEntity, ShouldResemble, testEntityID)
			})
			Convey("and fetch it back by entity", func() {
				shares, err := c.GetSharesByAuthor(ctx, testUserID, []entity.Entity{testEntityID})
				So(err, ShouldBeNil)
				So(shares.Items[0], ShouldNotBeNil)
				So(shares.Items[0].TargetEntity, ShouldResemble, testEntityID)
			})
			Convey("and fetch back something that doesn't exist", func() {
				missingEntity := entity.New("test_post", newTestUserID()+"_missing")
				shares, err := c.GetSharesByAuthor(ctx, testUserID, []entity.Entity{missingEntity})
				So(err, ShouldBeNil)
				So(len(shares.Items), ShouldEqual, 0)
			})
			Convey("and fetch summary", func() {
				shareSummary, err := c.GetSharesSummaries(ctx, []entity.Entity{testEntityID})
				So(err, ShouldBeNil)
				So(shareSummary.Items[0].ParentEntity, ShouldResemble, testEntityID)
				So(shareSummary.Items[0].Total, ShouldEqual, 1)
			})
			Convey("and fetch multiple back", func() {
				shares, err := c.GetShares(ctx, []string{share.ID})
				So(err, ShouldBeNil)
				So(len(shares.Items), ShouldEqual, 1)
				So(shares.Items[0].TargetEntity, ShouldResemble, testEntityID)
			})
			Convey("and fetch multiple missing should fail", func() {
				shares, err := c.GetShares(ctx, []string{share.ID + "_MISSING_"})
				So(err, ShouldBeNil)
				So(len(shares.Items), ShouldEqual, 0)
			})
			Convey("and delete it", func() {
				share2, err := c.DeleteShare(ctx, share.ID)
				So(err, ShouldBeNil)
				So(share2.TargetEntity, ShouldResemble, testEntityID)
				Convey("and it should be gone", func() {
					share3, err := c.DeleteShare(ctx, share.ID)
					So(err, ShouldBeNil)
					So(share3, ShouldBeNil)

					share3, err = c.GetShare(ctx, share.ID)
					So(err, ShouldBeNil)
					So(share3, ShouldBeNil)
				})
				Convey("and fetch it back by entity should be nil", func() {
					shares, err := c.GetSharesByAuthor(ctx, testUserID, []entity.Entity{testEntityID})
					So(err, ShouldBeNil)
					So(len(shares.Items), ShouldEqual, 0)
				})
				Convey("and fetch summary should have zero total shares", func() {
					shareSummary, err := c.GetSharesSummaries(ctx, []entity.Entity{testEntityID})
					So(err, ShouldBeNil)
					So(shareSummary.Items[0].ParentEntity, ShouldResemble, testEntityID)
					So(shareSummary.Items[0].Total, ShouldEqual, 0)
				})
			})
		})
	})
}

func TestClientIntegration_Reaction(t *testing.T) {
	t.Parallel()
	Convey("With integration client", t, func() {
		setup := newSetup()
		So(setup.Setup(), ShouldBeNil)
		Reset(func() {
			So(setup.Close(), ShouldBeNil)
		})
		ctx := setup.Context
		c := setup.Client
		testUserID := newTestUserID()

		Convey("Should be able to create, get, and delete a reaction", func() {
			parent := entity.New("test_parent", timestamp())
			testEmoteID := "Kappa"
			created, err := c.CreateReaction(ctx, parent, testUserID, testEmoteID)
			So(err, ShouldBeNil)
			So(created, ShouldNotBeNil)
			So(created.ParentEntity, ShouldResemble, parent)
			So(created.EmoteIDs, ShouldContain, testEmoteID)

			reaction, err := c.GetReactions(ctx, parent, testUserID)
			So(err, ShouldBeNil)
			So(reaction, ShouldNotBeNil)

			deleted, err := c.DeleteReaction(ctx, parent, testUserID, testEmoteID)
			So(err, ShouldBeNil)
			So(deleted, ShouldNotBeNil)
			So(len(deleted.EmoteIDs), ShouldEqual, 0)
		})
	})
}

func TestClientIntegration_Reaction_Summary(t *testing.T) {
	t.Parallel()
	Convey("With integration client", t, func() {
		setup := newSetup()
		So(setup.Setup(), ShouldBeNil)
		Reset(func() {
			So(setup.Close(), ShouldBeNil)
		})
		ctx := setup.Context
		c := setup.Client
		testUserID := newTestUserID()

		Convey("Should update the reaction summary when reactions are created and deleted", func() {
			parent := entity.New("test_parent", timestamp())

			c.CreateReaction(ctx, parent, testUserID, "kappa")
			s1, err := c.GetReactionsSummary(ctx, parent)
			So(err, ShouldBeNil)
			So(s1, ShouldNotBeNil)
			So(s1.Emotes, ShouldNotBeNil)
			So(s1.Emotes["kappa"], ShouldEqual, 1)

			c.DeleteReaction(ctx, parent, testUserID, "kappa")
			s2, err := c.GetReactionsSummary(ctx, parent)
			So(err, ShouldBeNil)
			So(s2, ShouldNotBeNil)
			So(len(s2.Emotes), ShouldEqual, 0)
		})

		Convey("Should get multiple reactions summaries by parent entity", func() {
			parent1 := entity.New("test_parent", timestamp())
			c1, err := c.CreateReaction(ctx, parent1, testUserID, "120")
			So(err, ShouldBeNil)
			So(c1, ShouldNotBeNil)

			parent2 := entity.New("test_parent", timestamp())
			c2, err := c.CreateReaction(ctx, parent2, testUserID, "121")
			So(err, ShouldBeNil)
			So(c2, ShouldNotBeNil)

			reactions, err := c.GetReactionsSummariesByParent(ctx, []entity.Entity{parent1, parent2}, nil)
			So(err, ShouldBeNil)
			So(reactions, ShouldNotBeNil)
			So(len(reactions.Items), ShouldEqual, 2)
			Convey("and get by user id", func() {
				options := duplo.GetReactionsSummariesByParentOptions{
					UserIDs: []string{testUserID},
				}
				reactions, err := c.GetReactionsSummariesByParent(ctx, []entity.Entity{parent1, parent2}, &options)
				So(err, ShouldBeNil)
				So(reactions, ShouldNotBeNil)
				So(len(reactions.Items), ShouldEqual, 2)
				for _, item := range reactions.Items {
					for _, sum := range item.EmoteSummaries {
						So(sum.UserIDs, ShouldResemble, []string{testUserID})
					}
				}
			})
		})

		Convey("Deleting non-existent reaction should return nil", func() {
			parent := entity.New("test:parent", timestamp())
			reactions, err := c.DeleteReaction(ctx, parent, testUserID, "kappa")
			So(reactions, ShouldBeNil)
			So(err, ShouldBeNil)
		})
	})
}

type TestSetup struct {
	context.Context
	*duplo.Client
	cancelFunc func()
	Common     service_common.ConfigCommon
}

func (t *TestSetup) Setup() error {
	t.Context, t.cancelFunc = context.WithTimeout(context.Background(), time.Second*10)
	if err := t.Common.Setup(); err != nil {
		return err
	}
	cfg := &duplo.Config{}
	if err := cfg.Load(t.Common.Config); err != nil {
		return err
	}
	t.Client = &duplo.Client{
		Config: cfg,
	}
	return nil
}

func (c *TestSetup) Close() error {
	c.cancelFunc()
	return c.Common.Close()
}
