package masonry

import (
	"fmt"
	"net/url"
	"strconv"

	"code.justin.tv/feeds/clients"
	"code.justin.tv/feeds/clients/feeddataflow"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/feeds-common/entity"
	"code.justin.tv/feeds/feeds-common/verb"
	"golang.org/x/net/context"
)

// Config configures masonry http client
type Config struct {
	Host func() string
}

// Load configuration information
func (c *Config) Load(dconf *distconf.Distconf) error {
	c.Host = dconf.Str("masonry.http_endpoint", "").Get
	return nil
}

// A Feed is a list of activities
type Feed struct {
	ID     string     `json:"id"`
	Cursor string     `json:"cursor"`
	Items  []Activity `json:"items"`
}

// Activity is the feed object that comes into fanout service
type Activity struct {
	Entity   entity.Entity          `json:"entity"`
	Verb     verb.Verb              `json:"verb"`
	Actor    entity.Entity          `json:"actor"`
	Metadata *feeddataflow.Metadata `json:"metadata,omitempty"`
}

type errResponse struct {
	StatusCode int    `json:"-"`
	Message    string `json:"message"`
}

func (e errResponse) Error() string {
	return fmt.Sprintf("%d: %s", e.StatusCode, e.Message)
}

// Client does normal HTTP requests to get information from masonry
type Client struct {
	RequestDoer    clients.RequestDoer
	NewHTTPRequest clients.NewHTTPRequest
	Config         *Config
}

func (c *Client) http(ctx context.Context, method, path string, queryParams url.Values, body interface{}, into interface{}) error {
	return clients.DoHTTP(ctx, c.RequestDoer, method, c.Config.Host()+path, queryParams, body, into, c.NewHTTPRequest)
}

// GetFeedOptions defines optional params when getting a feed
type GetFeedOptions struct {
	Limit  int
	Cursor string
}

// GetFeed returns a paginated feed
func (c *Client) GetFeed(ctx context.Context, feedID string, options *GetFeedOptions) (*Feed, error) {
	path := "/v1/feeds/" + feedID
	query := url.Values{}
	if options != nil {
		if options.Limit != 0 {
			query.Add("limit", strconv.Itoa(options.Limit))
		}
		if options.Cursor != "" {
			query.Add("cursor", options.Cursor)
		}
	}
	var ret Feed
	if err := c.http(ctx, "GET", path, query, nil, &ret); err != nil {
		return nil, err
	}
	return &ret, nil
}
