package entity

import (
	"encoding/json"
	"testing"

	. "github.com/smartystreets/goconvey/convey"
)

func TestEntity_Creation(t *testing.T) {
	Convey("Entity Creation", t, func() {
		for _, namespace := range []string{"", "test", "123"} {
			for _, id := range []string{"", "123", "a:123", "a:b:123", "a::123"} {
				Convey("With namespace["+namespace+"] id["+id+"]", func() {
					entity := New(namespace, id)
					So(entity, ShouldNotBeNil)
					So(entity.Namespace(), ShouldEqual, namespace)
					So(entity.ID(), ShouldEqual, id)

					parsedEntity, err := Decode(entity.Encode())
					So(err, ShouldBeNil)
					So(parsedEntity == entity, ShouldBeTrue)
					So(parsedEntity, ShouldResemble, entity)
				})
			}
		}
	})
}

func TestEntity_Empty(t *testing.T) {
	Convey("Default Entity", t, func() {
		var e Entity
		So(e.Namespace(), ShouldEqual, "")
		So(e.ID(), ShouldEqual, "")
	})
}

func TestEntity_Parsing(t *testing.T) {
	Convey("Entity Parsing", t, func() {
		Convey("Just ID, no namespace", func() {
			_, err := Decode("1234")
			So(err, ShouldNotBeNil)
		})
	})
}

func TestEntity_Encoding(t *testing.T) {
	Convey("Encode Multiple", t, func() {
		entities := []Entity{
			New("a", "123"),
			New("b", "456"),
			New("c", "789"),
		}
		encoded := EncodeAll(entities)
		So(encoded, ShouldHaveLength, 3)
		So(encoded[0], ShouldEqual, "a:123")
		So(encoded[1], ShouldEqual, "b:456")
		So(encoded[2], ShouldEqual, "c:789")
	})
}

func TestEntity_JSON(t *testing.T) {
	type Holder struct {
		Entity Entity `json:"entity"`
	}
	Convey("With entity holder", t, func() {
		Convey("Testing marshal/unmarshal", func() {
			entity := New("test", "123")

			a := Holder{Entity: entity}

			bytes, err := json.Marshal(a)
			So(err, ShouldBeNil)
			So(bytes, ShouldNotBeNil)

			b := Holder{}
			So(b, ShouldNotResemble, a)
			err = json.Unmarshal(bytes, &b)
			So(err, ShouldBeNil)
			So(b, ShouldResemble, a)
		})
		Convey("Testing backwards compatibility for marshal", func() {
			holder := Holder{
				Entity: New("test", "abc"),
			}
			buf, err := json.Marshal(holder)
			So(err, ShouldBeNil)
			So(string(buf), ShouldEqual, "{\"entity\":\"test:abc\"}")
		})
		Convey("Testing backwards compatibility for unmarshal", func() {
			bytes := []byte("{\"entity\":\"test:abc\"}")

			holder := Holder{}
			err := json.Unmarshal(bytes, &holder)
			So(err, ShouldBeNil)
			So(holder.Entity, ShouldResemble, New("test", "abc"))
		})
		Convey("Testing that the decode copies the value", func() {
			// See https://golang.org/pkg/encoding/#TextUnmarshaler
			bytes := []byte("{\"entity\": \"test:abc\"}")

			holder := Holder{}
			err := json.Unmarshal(bytes, &holder)
			So(err, ShouldBeNil)
			bytes[12] = 'T'
			So(holder.Entity, ShouldResemble, New("test", "abc"))
		})
	})

	type RefHolder struct {
		Entity *Entity `json:"entity"`
	}
	Convey("With entity ref holder", t, func() {
		Convey("Testing marshal/unmarshal", func() {
			entity := New("test", "123")

			a := RefHolder{Entity: &entity}

			bytes, err := json.Marshal(a)
			So(err, ShouldBeNil)
			So(bytes, ShouldNotBeNil)

			b := RefHolder{}
			So(b, ShouldNotResemble, a)
			err = json.Unmarshal(bytes, &b)
			So(err, ShouldBeNil)
			So(b, ShouldResemble, a)
		})
		Convey("Testing nil entity", func() {
			bytes := []byte("{}")

			holder := RefHolder{}
			err := json.Unmarshal(bytes, &holder)
			So(err, ShouldBeNil)
			So(holder.Entity, ShouldBeNil)
		})
		Convey("Testing backwards compatibility for marshal", func() {
			entity := New("test", "abc")
			holder := RefHolder{
				Entity: &entity,
			}
			buf, err := json.Marshal(holder)
			So(err, ShouldBeNil)
			So(string(buf), ShouldEqual, "{\"entity\":\"test:abc\"}")
		})
		Convey("Testing backwards compatibility for unmarshal", func() {
			bytes := []byte("{\"entity\":\"test:abc\"}")

			holder := RefHolder{}
			err := json.Unmarshal(bytes, &holder)
			So(err, ShouldBeNil)
			So(*holder.Entity, ShouldResemble, New("test", "abc"))
		})
	})
}
