package associations

import (
	"testing"

	. "github.com/smartystreets/goconvey/convey"
)

func init() {
	InitTestSchema()
}

func TestNew(t *testing.T) {

	Convey("When creating an association with a missing id", t, func() {
		e1 := Entity{Kind: UserKind}
		e2 := Entity{ID: "1234", Kind: UserKind}
		_, err := NewAssoc(e1, e2, "follows")
		Convey("err should be ErrInvalidID", func() {
			So(err, ShouldNotBeNil)
			So(err.Error(), ShouldContainSubstring, "Missing ID")
		})
	})

	Convey("When creating an association with a invalid ids", t, func() {
		e1 := Entity{ID: "☃", Kind: UserKind}
		e2 := Entity{ID: "1234", Kind: UserKind}
		_, err := NewAssoc(e1, e2, "follows")
		Convey("err should be ErrInvalidID", func() {
			So(err, ShouldNotBeNil)
			So(err.Error(), ShouldContainSubstring, "Invalid ID")
		})
	})

	Convey("When creating an association with an invalid type", t, func() {
		e1 := Entity{ID: "1234", Kind: UserKind}
		e2 := Entity{ID: "5678", Kind: UserKind}
		_, err := NewAssoc(e1, e2, "fail")
		Convey("err should be ErrInvalidKind", func() {
			So(err, ShouldNotBeNil)
			So(err.Error(), ShouldContainSubstring, "Invalid Kind")
		})
	})

	Convey("When creating an association with between identical entities", t, func() {
		e1 := Entity{ID: "1234", Kind: UserKind}
		e2 := Entity{ID: "1234", Kind: UserKind}
		_, err := NewAssoc(e1, e2, "follows")
		Convey("err should be ErrInvalidId", func() {
			So(err, ShouldNotBeNil)
			So(err.Error(), ShouldContainSubstring, "cannot be identical")
		})
	})

	Convey("When creating an association with e1 wrongly typed", t, func() {
		e1 := Entity{ID: "1234", Kind: TestKind}
		e2 := Entity{ID: "5678", Kind: UserKind}
		_, err := NewAssoc(e1, e2, "follows")
		Convey("err should be ErrInvalidKind", func() {
			So(err, ShouldNotBeNil)
			So(err.Error(), ShouldContainSubstring, "Invalid entity 1 kind")
		})
	})

	Convey("When creating an association with e2 wrongly typed", t, func() {
		e1 := Entity{ID: "1234", Kind: UserKind}
		e2 := Entity{ID: "5678", Kind: TestKind}
		_, err := NewAssoc(e1, e2, "follows")
		Convey("err should be ErrInvalidKind", func() {
			So(err, ShouldNotBeNil)
			So(err.Error(), ShouldContainSubstring, "Invalid entity 2 kind")
		})
	})

	Convey("When creating a valid association", t, func() {
		e1 := Entity{ID: "1234", Kind: UserKind}
		e2 := Entity{ID: "5678", Kind: UserKind}
		assoc, err := NewAssoc(e1, e2, "follows")
		Convey("err should be nil", func() {
			So(err, ShouldBeNil)
		})
		Convey("assoc should be a correctly initialized association", func() {
			So(assoc.E1, ShouldResemble, e1)
			So(assoc.E2, ShouldResemble, e2)
			So(assoc.Kind, ShouldResemble, FollowsKind)
		})
	})
}
