package v2

import (
	"fmt"
	"strings"

	"code.justin.tv/web/cohesion/rpc"

	"golang.org/x/net/context"

	"google.golang.org/grpc"
)

// Client provides an interface that defines the way clients should interact
// with Cohesion's server
type Client interface {
	BatchAssociations(context.Context, Entity, string, []string) (*ListAssocResponseWithMeta, error)
	Associations(context.Context, Entity, Entity) (*ListAssocResponseWithMeta, error)
	Create(context.Context, Entity, string, Entity, map[string]interface{}) error
	Delete(context.Context, Entity, string, Entity) error
	BulkDelete(context.Context, Entity, string, string) error
	Update(context.Context, Entity, string, Entity, string, map[string]interface{}) error
	BatchUpdate(context.Context, Entity, string, string, map[string]interface{}) error
	BulkUpdate(context.Context, Entity, string, string, string, map[string]interface{}) error
	Get(context.Context, Entity, string, Entity) (*ListResponse, error)
	BulkGet(context.Context, Entity, string, string, ...func(*rpc.RequestOptions)) (*ListResponse, error)
	Count(context.Context, Entity, string, string) (int, error)
	GetHitCounts(context.Context, string, int) (map[string]int, error)
}

type v2Client struct {
	conn   rpc.V2Client
	source string
}

// New returns a Client that will interact with the Cohesion server
// Contains nicer methods than the generated gRPC code
func New(serverAddr, source string) (Client, error) {
	conn, err := grpc.Dial(
		serverAddr,
		grpc.WithBlock(),
		grpc.WithInsecure(),
	)

	if err != nil {
		return nil, err
	}

	return &v2Client{
		conn:   rpc.NewV2Client(conn),
		source: fmt.Sprintf(source),
	}, nil
}

/**
 *  Functions for manipulating RequestOptions
 */

// Sort creates a function that converts a string to the rpc.Sort enum and sets
// it to the Sort field in rpc.RequestOptions
func Sort(sort string) func(*rpc.RequestOptions) {
	return func(opts *rpc.RequestOptions) {
		val, ok := rpc.Sort_value[strings.ToUpper(sort)]
		// if the all upper string is not a Sort enum, default to Sort_UNKNOWN
		if !ok {
			val = int32(0)
		}
		opts.Sort = rpc.Sort(val)
	}
}

// Limit creates a function that converts the int and sets it to the Limit field
// in rpc.RequestOptions
func Limit(limit int) func(*rpc.RequestOptions) {
	return func(opts *rpc.RequestOptions) {
		opts.Limit = int64(limit)
	}
}

// Offset creates a function that converts the int and sets it to the Offset
// field in rpc.RequestOptions
func Offset(offset int) func(*rpc.RequestOptions) {
	return func(opts *rpc.RequestOptions) {
		opts.Offset = int64(offset)
	}
}

// Cursor creates a function that sets the string to the Cursor field in
// rpc.RequestOptions
func Cursor(cursor string) func(*rpc.RequestOptions) {
	return func(opts *rpc.RequestOptions) {
		opts.Cursor = cursor
	}
}
