package loader

import (
	"database/sql"
	"strconv"
	"strings"
	"sync/atomic"
	"time"

	"code.justin.tv/feeds/clients/feed-settings"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/service-common"
	_ "github.com/lib/pq" // sql.Open("postgres", ...)
	"golang.org/x/net/context"
)

// Loader periodically copies data from audrey to feed-settings-loader
type Loader struct {
	Config             *Config
	FeedSettingsClient *feedsettings.Client
	AudreyRDS          *sql.DB
	stopFlag           int64

	Stats       *service_common.StatSender
	Log         log.Logger
	DebugLog    log.Logger
	PanicLogger service_common.PanicLogger
}

// SyncAll calls Sync until no unsynced Audrey data remains
func (l *Loader) SyncAll(ctx context.Context) error {
	limit := int(l.Config.RowCount.Get())
	offset := 0
	for atomic.LoadInt64(&l.stopFlag) == 0 {
		done, err := l.Sync(ctx, offset, limit)
		if err != nil {
			return err
		}
		if done {
			break
		}
		offset += limit
	}
	return nil
}

// Sync copies count rows from audrey into feed-settings-loader
func (l *Loader) Sync(ctx context.Context, offset, limit int) (done bool, retErr error) {
	tx, err := l.AudreyRDS.Begin()
	if err != nil {
		return false, err
	}
	defer func() {
		if retErr == nil {
			retErr = tx.Commit()
		} else {
			retErr = service_common.ConsolidateErrors([]error{retErr, tx.Rollback()})
		}
	}()

	l.Log.Log("count", limit, "Syncing batch")
	sqlStart := time.Now()
	rows, err := tx.Query(`
		SELECT
			user_id,
			subs_can_comment,
			friends_can_comment,
			user_disabled_comments,
			admin_disabled_comments
		FROM user_settings
		WHERE NOT (subs_can_comment AND friends_can_comment AND NOT user_disabled_comments AND NOT admin_disabled_comments) -- no need to copy untouched settings
		ORDER BY user_id ASC
		LIMIT $1 OFFSET $2
	`, limit, offset)
	l.Log.Log("duration", time.Since(sqlStart), "Fetched batch")
	if err != nil {
		return false, err
	}
	defer func() {
		retErr = service_common.ConsolidateErrors([]error{retErr, rows.Close()})
	}()

	migratedUserIDs := make([]string, 0, limit)
	defer func() {
		if len(migratedUserIDs) > 0 {
			l.Log.Log(
				"count", len(migratedUserIDs),
				"migratedUserIDs", strings.Join(migratedUserIDs, ","),
				"Loaded batch from Audrey",
			)
		}
	}()

	scanStart := time.Now()
	for rows.Next() {
		var userIDInt int
		params := &feedsettings.UpdateSettingsOptions{}
		if err := rows.Scan(
			&userIDInt,
			&params.SubsCanComment,
			&params.FriendsCanComment,
			&params.UserDisabledComments,
			&params.AdminDisabledComments,
		); err != nil {
			return false, err
		}
		userID := strconv.Itoa(userIDInt)

		itemStart := time.Now()
		_, err := l.FeedSettingsClient.UpdateSettings(ctx, "user:"+userID, params)
		l.Log.Log("duration", time.Since(itemStart), "Synced item from batch")
		if err != nil {
			return false, err
		}

		migratedUserIDs = append(migratedUserIDs, userID)
	}
	l.Log.Log("duration", time.Since(scanStart), "Synced batch")

	return len(migratedUserIDs) < limit, nil
}

// Setup creates the basic chitin context for the service
func (l *Loader) Setup() error {
	audreyRDS, err := sql.Open("postgres", l.Config.AudreyConnString.Get())
	if err != nil {
		return err
	}
	l.AudreyRDS = audreyRDS

	return nil
}

// Start the loader
func (l *Loader) Start() error {
	l.Log.Log("Loading data from Audrey")
	return l.SyncAll(context.Background())
}

// Close stops the loader
func (l *Loader) Close() error {
	atomic.StoreInt64(&l.stopFlag, 1)
	return nil
}
