package main

import (
	"os"

	"code.justin.tv/common/config"
	"code.justin.tv/feeds/clients/feed-settings"
	"code.justin.tv/feeds/feed-settings-loader/cmd/feed-settings-loader/internal/loader"
	"code.justin.tv/feeds/service-common"
)

const (
	teamName    = "feeds"
	serviceName = "feed-settings-loader"
)

// CodeVersion is set by build script
var CodeVersion string

var instance = service{
	osExit: os.Exit,
	serviceCommon: service_common.ServiceCommon{
		ConfigCommon: service_common.ConfigCommon{
			Team:    teamName,
			Service: serviceName,
		},
		CodeVersion: CodeVersion,
	},
}

type service struct {
	osExit        func(code int)
	sigChan       chan os.Signal
	serviceCommon service_common.ServiceCommon
	runner        service_common.ServiceRunner

	loader loader.Loader

	configs struct {
		LoaderConfig       loader.Config
		FeedSettingsConfig feedsettings.Config
	}
	skipGlobalConfigParse bool
}

func (f *service) setup() error {
	if err := f.serviceCommon.Setup(); err != nil {
		return err
	}
	if err := service_common.LoadConfigs(f.serviceCommon.Config, &f.configs.LoaderConfig, &f.configs.FeedSettingsConfig); err != nil {
		return err
	}
	if err := f.configs.LoaderConfig.LoadSecrets(f.serviceCommon.Secrets); err != nil {
		return err
	}

	if !f.skipGlobalConfigParse {
		if err := config.Parse(); err != nil {
			return err
		}
	}

	return nil
}

func (f *service) inject() {
	f.loader = loader.Loader{
		Config:             &f.configs.LoaderConfig,
		FeedSettingsClient: &feedsettings.Client{Config: &f.configs.FeedSettingsConfig},

		Log:         f.serviceCommon.Log,
		DebugLog:    f.serviceCommon.DebugLog,
		PanicLogger: f.serviceCommon.PanicLogger,
		Stats: &service_common.StatSender{
			SubStatter:   f.serviceCommon.Statsd.NewSubStatter("loader"),
			ErrorTracker: &f.serviceCommon.ErrorTracker,
		},
	}

	f.runner = service_common.ServiceRunner{
		Log: f.serviceCommon.Log,
		Services: []service_common.Service{
			&f.loader, &f.serviceCommon,
		},
		SigChan: f.sigChan,
	}
}

func (f *service) main() {
	if err := f.setup(); err != nil {
		if f.serviceCommon.Log != nil {
			f.serviceCommon.Log.Log("err", err, "Unable to load initial config")
		}
		service_common.SetupLogger.Log("err", err, "Unable to load initial config")
		f.osExit(1)
		return
	}
	f.inject()
	if err := f.runner.Execute(); err != nil {
		f.serviceCommon.Log.Log("err", err, "wait to end finished with an error")
		f.osExit(1)
		return
	}
	f.serviceCommon.Log.Log("Finished main")
}

func main() {
	instance.main()
}
