package clients

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io"
	"io/ioutil"
	"net/http"
	"net/url"

	"code.justin.tv/common/twitchhttp"
	"code.justin.tv/feeds/ctxlog"
	"golang.org/x/net/context"
)

const (
	maxBodySlurpSize      = int64(2 << 10)
	defaultStatSampleRate = 1.0
)

type errResponse struct {
	code int
	body bytes.Buffer
}

func (e *errResponse) Error() string {
	return fmt.Sprintf("invalid status code: %d: %s", e.code, e.body.String())
}

func (e *errResponse) HTTPCode() int {
	return e.code
}

func (e *errResponse) Body() *bytes.Buffer {
	return &e.body
}

func (e *errResponse) IsThrottled() bool {
	return e.code == http.StatusTooManyRequests
}

func httpCleanup(body io.ReadCloser, maxBodySlurpSize int64, retErr *error) {
	_, err2 := io.CopyN(ioutil.Discard, body, maxBodySlurpSize)
	if err2 != io.EOF && err2 != nil && *retErr == nil {
		*retErr = err2
	}
	err2 = body.Close()
	if err2 != nil && *retErr == nil {
		*retErr = err2
	}
}

// DoHTTP does common http stuff for getting a JSON response.  Using httpNewRequest so we can abstract out twitchhttp
// into clients that require it.
func do(ctx context.Context, method string, url string, queryParams url.Values, body interface{}, into interface{}, httpNewRequest func(string, string, io.Reader) (*http.Request, error), doFunc func(*http.Request) (*http.Response, error)) (retErr error) {
	var bodyReader io.Reader
	if body != nil {
		buf := &bytes.Buffer{}
		if err := json.NewEncoder(buf).Encode(&body); err != nil {
			return err
		}
		bodyReader = buf
	}
	req, err := httpNewRequest(method, url, bodyReader)
	if err != nil {
		return err
	}
	req = req.WithContext(ctx)
	if len(queryParams) > 0 {
		req.URL.RawQuery = queryParams.Encode()
	}

	resp, err := doFunc(req)
	if err != nil {
		return err
	}
	defer httpCleanup(resp.Body, maxBodySlurpSize, &retErr)
	if resp.StatusCode != http.StatusOK {
		errVal := &errResponse{code: resp.StatusCode}
		_, err2 := io.CopyN(&errVal.body, resp.Body, maxBodySlurpSize)
		if err2 != nil && err2 != io.EOF {
			return err2
		}
		return errVal
	}
	if into != nil {
		if err := json.NewDecoder(resp.Body).Decode(into); err != nil {
			return err
		}
	}
	return nil
}

// RequestDoer is anything that can issue HTTP requests.  Usually a http.Client
type RequestDoer interface {
	Do(req *http.Request) (*http.Response, error)
}

// DoHTTP does common http stuff using an http client for getting a JSON response
func DoHTTP(ctx context.Context, client RequestDoer, method string, host string, path string, queryParams url.Values, body interface{}, into interface{}) error {
	return do(ctx, method, host+path, queryParams, body, into, http.NewRequest, func(req *http.Request) (*http.Response, error) {
		return ctxlog.DoHTTPRequest(client, req)
	})
}

type twitchHTTPDoer struct {
	client twitchhttp.Client
	reqopt twitchhttp.ReqOpts
}

func (t *twitchHTTPDoer) Do(req *http.Request) (*http.Response, error) {
	return t.client.Do(req.Context(), req, t.reqopt)
}

// DoTwitchHTTP does http stuff using a twitch http client
func DoTwitchHTTP(ctx context.Context, client twitchhttp.Client, statName string, method string, path string, queryParams url.Values, body interface{}, reqOpts *twitchhttp.ReqOpts, into interface{}) error {
	combinedReqOpts := twitchhttp.MergeReqOpts(reqOpts, twitchhttp.ReqOpts{
		StatName:       statName,
		StatSampleRate: defaultStatSampleRate,
	})

	return do(ctx, method, path, queryParams, body, into, client.NewRequest, func(req *http.Request) (*http.Response, error) {
		return ctxlog.DoHTTPRequest(&twitchHTTPDoer{
			client: client,
			reqopt: combinedReqOpts,
		}, req)
	})
}
