package clients

import (
	"errors"
	"fmt"
	"io"
	"os"
	"time"

	"code.justin.tv/feeds/consulconf"
	"code.justin.tv/feeds/distconf"
	"github.com/hashicorp/consul/api"
)

// DefaultDistconf is a way to load a distconf for clients that don't already integrate with distconf
func DefaultDistconf() (*distconf.Distconf, io.Closer, error) {
	currentEnv := os.Getenv("ENVIRONMENT")
	if currentEnv == "" {
		currentEnv = "development"
	}
	consulClient := getConsulConnection()
	if consulClient == nil {
		return nil, nil, errors.New("unable to find a valid consul connection")
	}
	cconf := consulconf.Consulconf{
		KV:     consulClient.KV(),
		Prefix: fmt.Sprintf("settings/feeds/%s", currentEnv),
	}
	dconf := &distconf.Distconf{
		Readers: []distconf.Reader{&cconf},
	}
	refresher := &distconf.Refresher{
		ToRefresh: &cconf,
		WaitTime:  dconf.Duration("distconf.refresh_interval", time.Second*30),
	}
	go func() {
		err := refresher.Start()
		if err != nil {
			fmt.Println("Distconf refresher done.  Ignore this message if it is at service shutdown ", err)
		}
	}()
	return dconf, refresher, nil
}

func getConsulConnection() *api.Client {
	consulPriority := []string{"localhost:8500", "consul.internal.justin.tv"}
	for _, consulAddr := range consulPriority {
		config := api.DefaultConfig()
		config.Address = consulAddr
		config.Datacenter = "us-west2"
		client, err := api.NewClient(config)
		if err != nil {
			continue
		}
		if _, err = client.Agent().NodeName(); err != nil {
			continue
		}
		return client
	}
	return nil
}
