package consulconf

import (
	"fmt"
	"strings"

	"code.justin.tv/feeds/distconf"
	"github.com/hashicorp/consul/api"
)

// Consulconf allows fetching configuration from consul
type Consulconf struct {
	// KV should be your consul connection's KV struct
	KV *api.KV
	// Prefix is the key prefix inside consul
	Prefix string

	// QueryOptions are sent on every Get() operation
	QueryOptions *api.QueryOptions

	// WriteOptions are sent on every Write() operation
	WriteOptions *api.WriteOptions

	distconf.ReaderCache
}

// Get a key from consul and update watches if the key's value changed
func (s *Consulconf) Get(key string) ([]byte, error) {
	kvp, _, err := s.KV.Get(s.keyName(key), s.QueryOptions)
	if err != nil {
		return nil, err
	}
	var val []byte
	if kvp != nil {
		val = kvp.Value
	}
	distconf.ReaderCacheNotify(&s.ReaderCache, key, val)
	return val, nil
}

func (s *Consulconf) keyName(key string) string {
	return fmt.Sprintf("%s/%s", strings.Trim(s.Prefix, "/"), key)
}

// Write a new key to consul
func (s *Consulconf) Write(key string, value []byte) error {
	if value == nil {
		_, err := s.KV.Delete(s.keyName(key), s.WriteOptions)
		return err
	}
	kvp := &api.KVPair{
		Key:   s.keyName(key),
		Value: value,
	}

	_, err := s.KV.Put(kvp, s.WriteOptions)
	return err
}

// Refresh the entire cache, by doing a Get() on all stored keys
func (s *Consulconf) Refresh() {
	distconf.ReaderCacheRefresh(&s.ReaderCache, s, nil)
}

// Close does nothing and exists just to satisfy an interface
func (s *Consulconf) Close() {
}
