package ctxlog

import (
	"context"
	"math/rand"
	"net/http"
	"strconv"
	"sync"

	"code.justin.tv/feeds/log"
	goctx "golang.org/x/net/context"
)

// A Logger is anything that can log values
type Logger interface {
	Log(keyvals ...interface{})
}

var _ Logger = log.Logger(nil)

// LogWithDebugID is a logger that stores its DebugID for threading requests between servers
type LogWithDebugID struct {
	Logger  Logger
	DebugID string
}

// Log calls the embedded logger with the given params.  If LogWithDebugID is nil, safely does nothing.
func (l *LogWithDebugID) Log(paramsIn ...interface{}) {
	if l.Disabled() {
		return
	}
	params := make([]interface{}, 0, len(paramsIn)+2)
	params = append(params, "debug_id", l.DebugID)
	params = append(params, paramsIn...)
	l.Logger.Log(params...)
}

// Disabled returns true if this is nil or the embedded logger is disabled
func (l *LogWithDebugID) Disabled() bool {
	return l == nil || log.IsDisabled(l.Logger)
}

// HandlerC allows us to handle old style http request contexts
type HandlerC interface {
	ServeHTTPC(ctx goctx.Context, w http.ResponseWriter, r *http.Request)
}

// CtxHandler listens for HTTP requests and injects a debug ID if one is needed
type CtxHandler struct {
	NextC  HandlerC
	Logger Logger

	Header string
	IDGen  *rand.Rand
	mu     sync.Mutex
}

var _ http.Handler = &CtxHandler{}

func (c *CtxHandler) debugHeader() string {
	if c.Header == "" {
		return "X-Ctxlog-Debug"
	}
	return c.Header
}

func (c *CtxHandler) newDebugID() string {
	var randID int64
	if c.IDGen == nil {
		randID = rand.Int63()
	} else {
		c.mu.Lock()
		randID = c.IDGen.Int63()
		c.mu.Unlock()
	}
	return strconv.FormatInt(randID, 10)
}

// ServeHTTPC calls ServeHTTP but sets req's context to ctx, ignoring what's already there
func (c *CtxHandler) ServeHTTPC(ctx goctx.Context, rw http.ResponseWriter, req *http.Request) {
	c.ServeHTTP(rw, req.WithContext(ctx))
}

// ServeHTTP calls NextC with the correct ID injected into the current request
func (c *CtxHandler) ServeHTTP(rw http.ResponseWriter, req *http.Request) {
	currentDebugID := req.Header.Get(c.debugHeader())
	if currentDebugID != "" {
		if currentDebugID == "MakeNewID" {
			currentDebugID = c.newDebugID()
		}
		rw.Header().Set(c.debugHeader(), currentDebugID)
		req = req.WithContext(WithLogger(req.Context(), c.Logger, currentDebugID))
	}
	Log(req.Context(), func() {
		c.NextC.ServeHTTPC(req.Context(), rw, req)
	}, "http_path", req.URL.Path, "http_remote_addr", req.RemoteAddr)
}

// RequestDoer is anything that can issue HTTP requests
type RequestDoer interface {
	Do(req *http.Request) (*http.Response, error)
}

// Log calls a function, given a context, logging when the function starts and ends
func Log(ctx context.Context, f func(), params ...interface{}) {
	logger := DebugLogger(ctx)
	if !log.IsDisabled(logger) {
		l := log.NewContext(logger).With(params...)
		l.Log("phase", "start")
		defer l.Log("phase", "end")
	}
	f()
}

// DoHTTPRequest does an HTTP request, adding context information to the request.
func DoHTTPRequest(client RequestDoer, req *http.Request) (res *http.Response, err error) {
	Log(req.Context(), func() {
		dl := DebugLogger(req.Context())
		if dl != nil {
			req.Header.Set("X-Ctxlog-Debug", dl.DebugID)
		}
		res, err = client.Do(req)
	}, "destination", req.URL.String())
	return
}

type contextKey int

const (
	logKey contextKey = iota
)

// WithLogger returns a context that stores a request logger
func WithLogger(ctx context.Context, logger Logger, debugID string) context.Context {
	return context.WithValue(ctx, logKey, &LogWithDebugID{Logger: logger, DebugID: debugID})
}

// DebugLogger extracts a debug logger from a context.  If none is stored, it returns a nil LogWithDebugID which is safe
// to use.
func DebugLogger(ctx context.Context) *LogWithDebugID {
	currentVal := ctx.Value(logKey)
	if currentVal == nil {
		return nil
	}
	return currentVal.(*LogWithDebugID)
}
