package service_common

import (
	"fmt"
	"os"

	"time"

	"code.justin.tv/feeds/consulconf"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/errors"
	"github.com/hashicorp/consul/api"
)

// LoadableConfig is a distconf config struct that can load values from distconf
type LoadableConfig interface {
	Load(d *distconf.Distconf) error
}

// LoadConfigs loads an array of configs from distconf in parallel
func LoadConfigs(d *distconf.Distconf, configs ...LoadableConfig) error {
	errChan := make(chan error)
	for _, c := range configs {
		go func(c LoadableConfig) {
			errChan <- c.Load(d)
		}(c)
	}
	errs := make([]error, 0, len(configs))
	for i := 0; i < len(configs); i++ {
		errs = append(errs, <-errChan)
	}
	return ConsolidateErrors(errs)
}

// ConfigCommon loads just the config part of services
type ConfigCommon struct {
	Team          string
	Service       string
	Config        *distconf.Distconf
	Hostname      string
	Environment   string
	BaseDirectory string
	CustomReaders []distconf.Reader

	OsGetenv   func(string) string
	OsHostname func() (string, error)

	configRefresher distconf.Refresher
}

func (s *ConfigCommon) osGetenv(key string) string {
	if s.OsGetenv != nil {
		return s.OsGetenv(key)
	}
	return os.Getenv(key)
}

func (s *ConfigCommon) osHostname() (string, error) {
	if s.OsHostname != nil {
		return s.OsHostname()
	}
	return os.Hostname()
}

func (s *ConfigCommon) verify() error {
	if s.Team == "" {
		return errors.New("please set a Team field on ServiceCommon before calling setup")
	}
	if s.Service == "" {
		return errors.New("please set a Service field on ServiceCommon before calling setup")
	}
	return nil
}

func (s *ConfigCommon) setupHostname() error {
	if s.Hostname != "" {
		return nil
	}
	var err error
	s.Hostname, err = s.osHostname()
	if err != nil {
		s.Hostname = "_UNKNOWN_"
	}
	return nil
}

func (s *ConfigCommon) setupEnvironment() error {
	if s.Environment != "" {
		return nil
	}
	env := s.osGetenv("ENVIRONMENT")
	if env == "" {
		return errors.New("unable to find ENVIRONMENT variable in env")
	}
	s.Environment = env
	return nil
}

func (s *ConfigCommon) rootConfigs() ([]distconf.Reader, error) {
	configFilename := fmt.Sprintf("%sconfig/%s.json", s.BaseDirectory, s.Environment)
	cmdLineConfig := &distconf.CommandLine{Prefix: "--conf:"}
	envConfig := distconf.Env{
		Prefix:   s.Service + "_",
		OsGetenv: s.OsGetenv,
	}
	var baseConfig distconf.JSONConfig
	if err := baseConfig.RefreshFile(configFilename); err != nil {
		return nil, err
	}
	rootReaders := []distconf.Reader{cmdLineConfig, &envConfig, &baseConfig}
	ret := make([]distconf.Reader, 0, len(s.CustomReaders)+len(rootReaders))
	ret = append(ret, s.CustomReaders...)
	ret = append(ret, rootReaders...)
	return ret, nil
}

func (s *ConfigCommon) setupConfig() error {
	if s.Config != nil {
		return nil
	}
	rootConfigs, err := s.rootConfigs()
	if err != nil {
		return err
	}
	tmpDconf := distconf.Distconf{
		Readers: rootConfigs,
	}
	defer tmpDconf.Close()
	consulAddress := tmpDconf.Str("consul.address", "")
	consulDatacenter := tmpDconf.Str("consul.datacenter", "us-west2")
	if consulAddress.Get() == "" {
		return errors.New("unable to find consul.address location")
	}
	config := api.DefaultConfig()
	config.Address = consulAddress.Get()
	config.Datacenter = consulDatacenter.Get()

	client, err := api.NewClient(config)
	if err != nil {
		return err
	}

	if _, err = client.Agent().NodeName(); err != nil {
		return errors.Wrap(err, "unable to verify consul connection")
	}
	cconf1 := consulconf.Consulconf{
		KV:     client.KV(),
		Prefix: fmt.Sprintf("settings/%s/%s/%s", s.Team, s.Service, s.Environment),
	}
	cconf2 := consulconf.Consulconf{
		KV:     client.KV(),
		Prefix: fmt.Sprintf("settings/%s/%s", s.Team, s.Environment),
	}
	s.Config = &distconf.Distconf{
		Readers: append(rootConfigs, &cconf1, &cconf2),
		//Logger: func(key string, err error, msg string) {
		//	s.Log.Log("key", key, "err", err, msg)
		//},
	}
	s.configRefresher = distconf.Refresher{
		ToRefresh: distconf.ComboRefresher(s.refreshableReaders()),
		WaitTime:  s.Config.Duration("distconf.refresh_interval", time.Second*30),
	}
	return s.configRefresher.Setup()
}

// Setup ensures config can setup correctly
func (s *ConfigCommon) Setup() error {
	setups := []func() error{
		s.verify,
		s.setupHostname,
		s.setupEnvironment,
		s.setupConfig,
	}
	for _, setupFunc := range setups {
		err := setupFunc()
		if err != nil {
			return err
		}
	}
	return nil
}

// Close should be called when ConfigCommon is no longer needed
func (s *ConfigCommon) Close() error {
	errs := make([]error, 0, 2)
	if s.Config != nil {
		s.Config.Close()
		s.Config = nil
	}
	errs = append(errs, s.configRefresher.Close())
	return ConsolidateErrors(errs)
}

// Start allows periodic refreshing of config values
func (s *ConfigCommon) Start() error {
	return s.configRefresher.Start()
}

func (s *ConfigCommon) refreshableReaders() []distconf.Refreshable {
	ret := make([]distconf.Refreshable, 0, len(s.Config.Readers))
	for _, r := range s.Config.Readers {
		if refresher, ok := r.(distconf.Refreshable); ok {
			ret = append(ret, refresher)
		}

	}
	return ret
}
