package service_common

import (
	"expvar"
	"fmt"
	"net"
	"net/http"
	"net/http/pprof"
	"os"
	"strings"
	"sync"
	"sync/atomic"
	"time"

	"code.justin.tv/feeds/ctxlog/ctxloggoji"
	"code.justin.tv/feeds/distconf"
	"code.justin.tv/feeds/expvar2"
	"code.justin.tv/feeds/log"
	"code.justin.tv/feeds/rollbar"
	"code.justin.tv/feeds/rolllog"
	"code.justin.tv/feeds/sandyconf"
	"github.com/cactus/go-statsd-client/statsd"
	"goji.io"
	"goji.io/pat"
	"golang.org/x/net/context"
)

// SetupLogger is used before a logger can be created
var SetupLogger = log.NewLogfmtLogger(os.Stdout, log.Discard)

// ServiceCommon is the root structure for all feeds services and sets up commonly needed things.
type ServiceCommon struct {
	ConfigCommon

	hasFinished bool
	stopChannel chan struct{}
	wg          sync.WaitGroup

	Secrets       *distconf.Distconf
	Log           log.Logger
	DebugLog      log.Logger
	Statsd        statsd.Statter
	ErrorTracker  ErrorTracker
	ExpvarHandler expvar2.Handler
	DebugMutex    *http.ServeMux
	CodeVersion   string
	Stats         Stats
	PanicLogger   PanicLogger

	gometrics Gometrics

	debugListener net.Listener

	setupSync    sync.RWMutex
	lastSetupErr error
}

// Stats keep track of common stats during normal service common operation
type Stats struct {
	RollbarErrors int64
}

// CreateDefaultMux creates a goji mux that has /debug/running and a logging middleware
func CreateDefaultMux(Log log.Logger, DebugLog log.Logger, PanicLogger PanicLogger) *goji.Mux {
	mux := goji.NewMux()
	logMiddleware := HTTPLoggerMiddleware{
		Log:         Log,
		DebugLogger: DebugLog,
		PanicLogger: PanicLogger,
	}
	mux.Handle(pat.Get("/debug/running"), &HTTPStatusOk{
		Log: Log,
	})
	mux.Use(logMiddleware.NewHandler)
	mux.UseC(ctxloggoji.GojiMiddleware(Log))
	return mux
}

// CreateDefaultMux creates a default mux that works for most twitch HTTP applications
func (s *ServiceCommon) CreateDefaultMux() *goji.Mux {
	return CreateDefaultMux(s.Log, s.DebugLog, s.PanicLogger)
}

func (s *ServiceCommon) Format(f fmt.State, c rune) {
	fmt.Fprintf(f, "Service common: (%s)", s.Environment)
}

// Setup returns non nil if the service can setup itself correctly
func (s *ServiceCommon) Setup() (retErr error) {
	s.setupSync.Lock()
	defer s.setupSync.Unlock()
	if s.hasFinished {
		return s.lastSetupErr
	}
	defer func() {
		s.hasFinished = true
		s.lastSetupErr = retErr
	}()
	// For log messages that happen during setup, before a real logger is set
	if s.Log == nil {
		s.Log = SetupLogger
		defer func() {
			if s.Log == SetupLogger {
				s.Log = nil
			}
		}()
	}
	s.stopChannel = make(chan struct{})
	setups := []func() error{
		s.ConfigCommon.Setup,
		s.setupSecrets,
		s.setupLogging,
		s.setupStatsd,
		s.setupGometrics,
		s.setupExvarHandler,
		s.setupDebugPort,
	}
	for _, setupFunc := range setups {
		err := setupFunc()
		if err != nil {
			return err
		}
	}
	s.Log.Log("code_version", s.CodeVersion, "pid", os.Getpid(), "Common setup completed")
	if s.Secrets == nil {
		s.Log.Log("No sandstorm setup.  Could not find correct ARN")
	}
	return nil
}

// Close ends any listening ports and goroutines the service created
func (s *ServiceCommon) Close() error {
	close(s.stopChannel)
	errs := make([]error, 0, 4)
	if s.debugListener != nil {
		errs = append(errs, s.debugListener.Close())
		s.debugListener = nil
	}
	if s.Statsd != nil {
		errs = append(errs, s.Statsd.Close())
		s.Statsd = nil
	}
	errs = append(errs, s.ConfigCommon.Close())
	if s.Secrets != nil {
		s.Secrets.Close()
		s.Secrets = nil
	}
	s.wg.Wait()
	return ConsolidateErrors(errs)
}

func (s *ServiceCommon) setupSecrets() error {
	if s.Secrets != nil {
		return nil
	}
	arn := s.Config.Str("sandstorm.arn", "").Get()
	if arn == "" {
		s.Log.Log("No sandstorm ARN set.  Not setting up sandstorm secrets")
		s.Secrets = &distconf.Distconf{
			Logger: func(key string, err error, msg string) {
				s.Log.Log("key", key, "err", err, msg)
			},
		}
		return nil
	}
	mc := sandyconf.ManagerConstructor{}
	sandManager := mc.Manager(arn)

	sandstormConf := sandyconf.Sandyconf{
		Team:        s.Team,
		Service:     s.Service,
		Environment: s.Environment,
		Manager:     sandManager,
	}

	rootConfigs, err := s.rootConfigs()
	if err != nil {
		return err
	}

	s.Secrets = &distconf.Distconf{
		Readers: append(rootConfigs, &sandstormConf),
		Logger: func(key string, err error, msg string) {
			s.Log.Log("key", key, "err", err, msg)
		},
	}
	s.configRefresher.ToRefresh = append(
		s.configRefresher.ToRefresh.(distconf.ComboRefresher),
		&sandstormConf,
	)

	return nil
}

func (s *ServiceCommon) setupLogging() error {

	if s.Log != nil && s.Log != SetupLogger {
		// Logging already set.  Don't bother setting it up
		return nil
	}
	// Set a temp logger for errors getting config or secrets setting up logging
	rollbarConfig := rolllog.Config{
		// Get the access token from secrets, not regular config
		AccessToken: s.Secrets.Str("rollbar.access_token", ""),
		SendTimeout: s.Config.Duration("rollbar.send_timeout", time.Second*3),
	}

	clientDefaults := rollbar.DataOptionals{}
	clientDefaults.MergeFrom(&rollbar.DefaultConfiguration)
	clientDefaults.CodeVersion = s.CodeVersion

	rollbarClient := &rollbar.Client{
		Environment:     s.Environment,
		MessageDefaults: &clientDefaults,
	}

	rlog := &rolllog.Rolllog{
		OnErr: func(err error) {
			atomic.AddInt64(&s.Stats.RollbarErrors, 1)
		},
		Client:          rollbarClient,
		DefaultErrLevel: rollbar.Error,
		DefaultMsgLevel: rollbar.Info,
	}
	rollbarConfig.Monitor(rlog)
	stdoutLog := log.NewLogfmtLogger(os.Stdout, log.Discard)
	gatedStdout := &log.Gate{
		DisabledFlag: 1,
		Logger:       stdoutLog,
	}
	multiLogger := log.MultiLogger([]log.Logger{
		rlog, gatedStdout,
	})
	chanLogger := &log.ChannelLogger{
		Out:    make(chan []interface{}, 64),
		OnFull: log.Discard,
	}
	s.wg.Add(1)
	go func() {
		defer s.wg.Done()
		log.DrainChannel(multiLogger, chanLogger.Out, s.stopChannel)
	}()
	stdoutEnabled := s.Config.Bool("logging.to_stdout", false)
	if stdoutEnabled.Get() {
		s.PanicLogger = &PanicDumpWriter{
			Output: os.Stderr,
		}
	}
	watch := func() {
		if stdoutEnabled.Get() {
			gatedStdout.Enable()
		} else {
			gatedStdout.Disable()
		}
	}
	watch()
	stdoutEnabled.Watch(watch)
	s.Log = &StackLogger{
		RootLogger: chanLogger,
	}

	debugLog := &log.Gate{
		DisabledFlag: 1,
		Logger:       s.Log,
	}
	debugEnabled := s.Config.Bool("logging.debug_enabled", false)
	watch2 := func() {
		if debugEnabled.Get() {
			s.Log.Log("Debug logging turned on")
			debugLog.Enable()
		} else {
			s.Log.Log("Debug logging turned off")
			debugLog.Disable()
		}
	}
	watch2()
	debugEnabled.Watch(watch2)
	s.DebugLog = log.NewContext(debugLog).With("level", rollbar.Debug)
	s.DebugLog.Log("Debug log setup complete")
	return nil
}

func sanitizeStatsd(s string) string {
	if len(s) > 32 {
		s = s[0:32]
	}
	return strings.Replace(s, ".", "_", -1)
}

func (s *ServiceCommon) setupStatsd() error {
	if s.Statsd != nil {
		return nil
	}

	var err error
	statsdHostport := s.Config.Str("statsd.hostport", "").Get()
	if statsdHostport == "" {
		s.Log.Log("No statsd hostport setup.  Not sending metrics")
		s.Statsd, err = statsd.NewNoopClient()
		return err
	}
	flushInterval := s.Config.Duration("statsd.flush_interval", time.Second).Get()
	flushBytes := s.Config.Int("statsd.flush_bytes", 512).Get()

	prefix := fmt.Sprintf("%s.%s.%s.%s", sanitizeStatsd(s.Team), sanitizeStatsd(s.Service), sanitizeStatsd(s.Environment), sanitizeStatsd(s.Hostname))
	s.Statsd, err = statsd.NewBufferedClient(statsdHostport, prefix, flushInterval, int(flushBytes))
	s.Log.Log("addr", statsdHostport, "prefix", prefix, "Statsd setup complete")
	return err
}

func (s *ServiceCommon) setupGometrics() error {
	s.ErrorTracker = ErrorTracker{
		Log:      s.Log,
		DebugLog: s.DebugLog,
	}
	s.gometrics = Gometrics{
		Stats: &StatSender{
			SubStatter:   s.Statsd.NewSubStatter("go"),
			ErrorTracker: &s.ErrorTracker,
		},
		StartTime: time.Now(),
	}
	s.wg.Add(1)
	go func() {
		defer s.wg.Done()
		s.gometrics.Start(s.stopChannel, s.Config.Duration("gostats.collect_interval", time.Second*5))
	}()

	return nil
}

var startTime = time.Now()

func (s *ServiceCommon) setupExvarHandler() error {
	if s.ExpvarHandler.Exported == nil {
		s.ExpvarHandler.Exported = make(map[string]expvar.Var)
	}
	if s.ExpvarHandler.Logger == nil {
		s.ExpvarHandler.Logger = func(err error) {
			s.Log.Log("err", err)
		}
	}
	s.ExpvarHandler.Exported["env"] = expvar2.EnviromentalVariables()
	s.ExpvarHandler.Exported["conf"] = s.Config.Var()
	s.ExpvarHandler.Exported["untracked_errors"] = s.ErrorTracker.Var()
	s.ExpvarHandler.Exported["uptime"] = expvar.Func(func() interface{} {
		return time.Since(startTime).String()
	})
	codeVersionExpvar := expvar.String{}
	codeVersionExpvar.Set(s.CodeVersion)
	s.ExpvarHandler.Exported["code_version"] = &codeVersionExpvar
	if s.Secrets != nil {
		s.ExpvarHandler.Exported["secrets"] = expvar.Func(func() interface{} {
			return s.Secrets.Keys()
		})
	}
	return nil
}

func (s *ServiceCommon) setupDebugPort() error {
	mux := http.NewServeMux()
	d := http.Server{
		Handler: mux,
	}

	mux.Handle("/debug/vars", &s.ExpvarHandler)
	mux.Handle("/debug/pprof/", http.HandlerFunc(pprof.Index))
	mux.Handle("/debug/pprof/cmdline", http.HandlerFunc(pprof.Cmdline))
	mux.Handle("/debug/pprof/profile", http.HandlerFunc(pprof.Profile))
	mux.Handle("/debug/pprof/symbol", http.HandlerFunc(pprof.Symbol))
	mux.Handle("/debug/pprof/trace", http.HandlerFunc(pprof.Trace))

	var err error
	s.debugListener, err = RetriedListen(context.Background(), s.Config.Str("debug.addr", ":6060").Get(), time.Second*30, time.Second)
	if err != nil {
		return err
	}
	go func() {
		err := d.Serve(s.debugListener)
		s.Log.Log(log.Err, err, "Finished serving on debug port")
	}()
	return nil
}
